<?php
// profile.php (Teacher Profile View/Update)

// Start Output Buffering (Good Practice)
ob_start();
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Ensure session is started
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

// ---------------------------------------------------------------------
// --- 1. AUTHENTICATION & INITIAL FETCH ---
// ---------------------------------------------------------------------
// Check for the consistent teacher ID session key
if (!isset($_SESSION['teacher_id']) || empty($_SESSION['teacher_id'])) {
    if (ob_get_level() > 0) ob_end_clean();
    header("Location: teacher_login.php");
    exit();
}

// Database Connection Details (Using provided details)
$host = "localhost";
$user = "thevkin_schlteachdb";
$pass = "VK_AutSchlDb";
$db   = "thevkin_schlteachdb";

// Create database connection
$conn = mysqli_connect($host, $user, $pass, $db);

if (!$conn) {
    // In a real application, log this error and show a generic message.
    die("Database Connection failed. Please try again later.");
}

$msg = "";
$msg_type = "";

// The ID of the currently logged-in Teacher (Assume it's a string/varchar, using 's' for consistency)
$logged_in_id = $_SESSION['teacher_id'];

// Initial Fetch of Data (using Prepared Statements)
$stmt_fetch = $conn->prepare("SELECT * FROM teacher_profiles WHERE id = ?");
$stmt_fetch->bind_param("s", $logged_in_id); // 's' for string/varchar ID
$stmt_fetch->execute();
$result_fetch = $stmt_fetch->get_result();
$user_data = $result_fetch->fetch_assoc();
$stmt_fetch->close();


// If the fetch fails for some reason (user ID does not exist), redirect to logout
if (!$user_data) {
    if (ob_get_level() > 0) ob_end_clean();
    header("Location: logout_teacher.php");
    exit();
}

// --- Display variables based on fetched data ---
$display_name = htmlspecialchars($user_data['first_name'] . ' ' . $user_data['last_name']);
$path = (!empty($user_data['photo']) && file_exists($user_data['photo'])) ? $user_data['photo'] : "uploads/default.png";

// ---------------------------------------------------------------------
// --- 2. UPDATE PROFILE LOGIC ---
// ---------------------------------------------------------------------
if (isset($_POST['update_btn'])) {
    
    // Sanitize and trim inputs
    $first_name = trim($_POST['first_name']);
    $last_name  = trim($_POST['last_name']);
    $username   = trim($_POST['username']);
    $email      = trim($_POST['email']);
    $phone      = trim($_POST['phone']);
    $password_raw = trim($_POST['password']);
    
    $new_photo_path = $user_data['photo'];
    $upload_error = false;
    $has_password_update = !empty($password_raw);

    // --- PHOTO UPLOAD LOGIC ---
    if(isset($_FILES['photo']['name']) && $_FILES['photo']['name'] != ""){
        if (!file_exists('uploads')) {
            mkdir('uploads', 0755, true);
        }
        
        $upload_dir = 'uploads/';
        $filename = $_FILES['photo']['name'];
        $file_tmp = $_FILES['photo']['tmp_name'];
        $file_ext = strtolower(pathinfo($filename, PATHINFO_EXTENSION));
        $valid_extensions = ["jpg", "jpeg", "png", "gif"];
        $new_filename = $upload_dir . uniqid() . "." . $file_ext;

        if (in_array($file_ext, $valid_extensions) && @getimagesize($file_tmp)) {
            if (move_uploaded_file($file_tmp, $new_filename)) {
                $new_photo_path = $new_filename;
                
                // Delete old photo (if it's not the default one)
                $old_photo = $user_data['photo'];
                if (!empty($old_photo) && $old_photo != "uploads/default.png" && file_exists($old_photo)) {
                    @unlink($old_photo);
                }
            } else {
                $msg = "Error uploading file.";
                $msg_type = "error";
                $upload_error = true;
            }
        } else {
            $msg = "Invalid file or file type. Only JPG, PNG, GIF allowed.";
            $msg_type = "error";
            $upload_error = true;
        }
    }

    // --- DUPLICATE USERNAME CHECK ---
    if(!$upload_error) {
        $stmt_check = $conn->prepare("SELECT id FROM teacher_profiles WHERE username = ? AND id != ?");
        $stmt_check->bind_param("ss", $username, $logged_in_id);
        $stmt_check->execute();
        $result_check = $stmt_check->get_result();

        if ($result_check->num_rows > 0) {
            $msg = "Error: The username '{$username}' is already taken by another teacher.";
            $msg_type = "error";
            $upload_error = true; // Block the update
        }
        $stmt_check->close();
    }
    
    // --- DATABASE UPDATE LOGIC ---
    if(!$upload_error) {
        
        $bind_params = [];
        $bind_types = "";
        
        if ($has_password_update) {
            // HASH THE PASSWORD
            $password_hashed = password_hash($password_raw, PASSWORD_DEFAULT);
            
            // Query with password
            $update_sql = "
                UPDATE teacher_profiles SET 
                    first_name = ?,
                    last_name = ?,
                    username = ?,
                    email = ?,
                    phone = ?,
                    photo = ?,
                    password = ?
                WHERE id = ?
            ";
            // ssssssss: All fields are assumed to be string/varchar, including the ID for consistency
            $bind_types = "ssssssss"; 
            $bind_params = [&$first_name, &$last_name, &$username, &$email, &$phone, &$new_photo_path, &$password_hashed, &$logged_in_id];

        } else {
             // Query WITHOUT password
             $update_sql = "
                UPDATE teacher_profiles SET 
                    first_name = ?,
                    last_name = ?,
                    username = ?,
                    email = ?,
                    phone = ?,
                    photo = ?
                WHERE id = ?
            ";
            // sssssss: All fields are assumed to be string/varchar, including the ID for consistency
            $bind_types = "sssssss";
            $bind_params = [&$first_name, &$last_name, &$username, &$email, &$phone, &$new_photo_path, &$logged_in_id];
        }

        $stmt = $conn->prepare($update_sql);
        
        if ($stmt === false) {
             $msg = "Database Prepare Error: " . $conn->error;
             $msg_type = "error";
             $upload_error = true;
        }

        if(!$upload_error) {
            // Bind parameters using call_user_func_array
            // array_unshift inserts the $bind_types string at the beginning of $bind_params
            array_unshift($bind_params, $bind_types);
            
            // This line ensures the parameters are passed by reference to bind_param
            if (!call_user_func_array([$stmt, 'bind_param'], $bind_params)) {
                 $msg = "Bind Param Error: " . $stmt->error;
                 $msg_type = "error";
            }
            
            if($stmt->execute()) {
                $msg = "Teacher Profile Updated Successfully!";
                $msg_type = "success";
                
                // --- CRITICAL STEP 1: REFRESH SESSION DATA AFTER SUCCESSFUL UPDATE ---
                // Re-fetch the data to update $user_data and session variables
                $stmt_refresh = $conn->prepare("SELECT * FROM teacher_profiles WHERE id = ?");
                $stmt_refresh->bind_param("s", $logged_in_id);
                $stmt_refresh->execute();
                $result_refresh = $stmt_refresh->get_result();
                $refreshed_data = $result_refresh->fetch_assoc();
                $stmt_refresh->close();

                if ($refreshed_data) {
                    // Update session variables
                    $_SESSION['teacher_name'] = $refreshed_data['first_name'] . ' ' . $refreshed_data['last_name'];
                    $_SESSION['teacher_photo'] = $refreshed_data['photo'];
                }

                // Re-fetch the current data for display on THIS iframe page after successful update
                $user_data = $refreshed_data;
                $display_name = htmlspecialchars($user_data['first_name'] . ' ' . $user_data['last_name']);
                $path = (!empty($user_data['photo']) && file_exists($user_data['photo'])) ? $user_data['photo'] : "uploads/default.png";
                
                // You can add a success notification here if not using postMessage
                // e.g., echo "<script>alert('Profile Updated!');</script>";
                
            } else {
                $msg = "Error updating profile: " . $stmt->error;
                $msg_type = "error";
            }
            $stmt->close();
        }
    }
    
    // Send message to parent frame only if a message exists
    if ($msg != "") {
        $js_msg = htmlspecialchars($msg, ENT_QUOTES, 'UTF-8');
        $js_type = htmlspecialchars($msg_type, ENT_QUOTES, 'UTF-8');
        echo "<script>
            window.parent.postMessage({
                action: 'show_alert',
                message: '{$js_msg}',
                type: '{$js_type}'
            }, '*');
        </script>";
    }
}

mysqli_close($conn);

// ---------------------------------------------------------------------
// --- 3. DISPLAY LOGIC (HTML START) ---
// ---------------------------------------------------------------------
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Edit Teacher Profile</title>
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        :root {
            --primary-color: #1a56db;
            --secondary-color: #172554;
            --accent-color: #f72585;
            --light-color: #f8f9fa;
            --dark-color: #212529;
            --success-color: #2ecc71;
            --danger-color: #e74c3c;
            --border-radius: 12px;
            --box-shadow: 0 10px 30px rgba(0, 0, 0, 0.1);
            --transition: all 0.3s ease;
        }

        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body { 
            background: #f1f5f9; 
            min-height: 100vh; 
            padding-bottom: 20px; 
        }
        
        .container {
            width: 100%;
            max-width: 900px;
            background: rgba(255, 255, 255, 0.95);
            border-radius: var(--border-radius);
            box-shadow: var(--box-shadow);
            overflow: hidden;
            display: flex;
            flex-direction: column;
            margin: 20px auto; 
            animation: fadeIn 0.5s ease-in-out;
        }

        @keyframes fadeIn {
            from { opacity: 0; transform: translateY(20px); }
            to { opacity: 1; transform: translateY(0); }
        }

        .profile-header {
            background: linear-gradient(135deg, var(--primary-color), var(--secondary-color));
            color: white;
            padding: 2.5rem;
            text-align: center;
            position: relative;
            overflow: hidden;
        }

        .profile-header h1 {
            font-size: 2.2rem;
            margin-bottom: 0.5rem;
            position: relative;
            z-index: 1;
        }

        .profile-header p {
            opacity: 0.9;
            position: relative;
            z-index: 1;
        }

        .profile-form {
            padding: 2.5rem;
        }

        .form-grid {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 1.5rem;
        }

        .form-group {
            margin-bottom: 1.5rem;
            position: relative;
        }

        .form-group.full-width {
            grid-column: 1 / -1;
        }

        .form-group label {
            display: block;
            margin-bottom: 0.5rem;
            font-weight: 600;
            color: var(--dark-color);
            font-size: 0.95rem;
        }

        .form-group label i {
            margin-right: 0.5rem;
            color: var(--primary-color);
        }

        .form-group input {
            width: 100%;
            padding: 0.875rem 1rem 0.875rem 2.5rem;
            border: 2px solid #e1e5eb;
            border-radius: 8px;
            font-size: 1rem;
            transition: var(--transition);
            background-color: rgba(255, 255, 255, 0.9);
        }

        .form-group input:focus {
            outline: none;
            border-color: var(--primary-color);
            box-shadow: 0 0 0 3px rgba(26, 86, 219, 0.1);
        }

        .form-group .input-icon {
            position: absolute;
            left: 1rem;
            top: 2.6rem;
            color: #6c757d;
        }

        .photo-upload {
            display: flex;
            align-items: center;
            gap: 1.5rem;
        }

        .current-photo {
            width: 120px;
            height: 120px;
            border-radius: 50%;
            object-fit: cover;
            border: 4px solid var(--light-color);
            box-shadow: 0 8px 20px rgba(0, 0, 0, 0.15);
            transition: var(--transition);
        }

        .upload-btn {
            display: inline-block;
            padding: 0.875rem 1.5rem;
            background: linear-gradient(135deg, var(--primary-color), var(--secondary-color));
            color: white;
            border-radius: 8px;
            cursor: pointer;
            transition: var(--transition);
            text-align: center;
            font-weight: 500;
            box-shadow: 0 4px 10px rgba(26, 86, 219, 0.3);
        }
        .upload-btn:hover {
             opacity: 0.9;
        }

        .form-group input[type="file"] {
            display: none;
        }

        .btn {
            display: inline-block;
            padding: 0.875rem 2rem;
            background: linear-gradient(135deg, var(--primary-color), var(--secondary-color));
            color: white;
            border: none;
            border-radius: 8px;
            font-size: 1rem;
            font-weight: 600;
            cursor: pointer;
            transition: var(--transition);
            text-align: center;
            box-shadow: 0 4px 15px rgba(26, 86, 219, 0.3);
        }

        .btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(26, 86, 219, 0.4);
        }

        .btn-container {
            display: flex;
            justify-content: flex-end;
            margin-top: 2rem;
            grid-column: 1 / -1;
        }

        .msg {
            padding: 1rem 1.5rem;
            margin-bottom: 1.5rem;
            border-radius: 8px;
            display: flex;
            align-items: center;
            gap: 0.75rem;
            animation: slideIn 0.3s ease-out;
        }

        /* Message styling (add these, as they were only in the style attribute before) */
        .msg.error {
            background-color: rgba(231, 76, 60, 0.1);
            color: var(--danger-color);
            border: 1px solid rgba(231, 76, 60, 0.3);
        }
        .msg.success {
            background-color: rgba(46, 204, 113, 0.1);
            color: var(--success-color);
            border: 1px solid rgba(46, 204, 113, 0.3);
        }

        /* ----------------------------------------------- */
        /* --- MOBILE IMPROVEMENTS (max-width: 768px) --- */
        /* ----------------------------------------------- */
        @media (max-width: 768px) {
            .container {
                margin: 0; 
                border-radius: 0; 
                box-shadow: none;
            }
            .profile-header {
                 padding: 2rem 1.5rem 2rem; 
            }
            .profile-form {
                 padding: 1.5rem;
            }
            .form-grid {
                grid-template-columns: 1fr; 
                gap: 1rem; 
            }
            .form-group {
                margin-bottom: 1rem; 
            }
            .form-group.full-width {
                grid-column: auto; 
            }
            .photo-upload {
                flex-direction: column;
                text-align: center;
                gap: 1rem;
                margin-top: 0.5rem; 
            }
            .current-photo {
                width: 100px; 
                height: 100px;
                order: -1; 
            }
            .upload-btn {
                width: 100%; 
                max-width: 250px;
            }
            .btn-container {
                justify-content: center;
                margin-top: 1.5rem;
                grid-column: auto;
            }
            .btn {
                width: 100%;
                max-width: 300px; 
                padding: 1rem 1rem; 
            }
            .form-group .input-icon {
                top: 2.5rem; 
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="profile-header">
            <h1>Edit Teacher Profile</h1> <p><?php echo $display_name; ?>. Update your information</p>
        </div>
        
        <div class="profile-form">
            <?php if($msg != "" && $msg_type == "error"): ?>
                <div class='msg error'>
                    <i class='fas fa-exclamation-circle'></i>
                    <?php echo htmlspecialchars($msg); ?>
                </div>
            <?php elseif($msg != "" && $msg_type == "success"): ?>
                <div class='msg success'>
                    <i class='fas fa-check-circle'></i>
                    <?php echo htmlspecialchars($msg); ?>
                </div>
            <?php endif; ?>
            
            <form method="POST" enctype="multipart/form-data">
                <div class="form-grid">
                    
                    <div class="form-group">
                        <label for="first_name"><i class="fas fa-user"></i> First Name</label>
                        <i class="fas fa-user input-icon"></i>
                        <input type="text" id="first_name" name="first_name" value="<?php echo htmlspecialchars($user_data['first_name'] ?? ''); ?>" required>
                    </div>
                    
                    <div class="form-group">
                        <label for="last_name"><i class="fas fa-user-tag"></i> Last Name</label>
                        <i class="fas fa-user-tag input-icon"></i>
                        <input type="text" id="last_name" name="last_name" value="<?php echo htmlspecialchars($user_data['last_name'] ?? ''); ?>" required>
                    </div>

                    <div class="form-group full-width">
                        <label for="email"><i class="fas fa-envelope"></i> Email Address</label>
                        <i class="fas fa-envelope input-icon"></i>
                        <input type="email" id="email" name="email" value="<?php echo htmlspecialchars($user_data['email'] ?? ''); ?>" required>
                    </div>
                    
                    <div class="form-group">
                        <label for="username"><i class="fas fa-id-badge"></i> Username</label>
                        <i class="fas fa-id-badge input-icon"></i>
                        <input type="text" id="username" name="username" value="<?php echo htmlspecialchars($user_data['username'] ?? ''); ?>" required>
                    </div>

                    <div class="form-group">
                        <label for="phone"><i class="fas fa-phone"></i> Phone Number</label>
                        <i class="fas fa-phone input-icon"></i>
                        <input type="text" id="phone" name="phone" value="<?php echo htmlspecialchars($user_data['phone'] ?? ''); ?>" required>
                    </div>

                    <div class="form-group full-width">
                        <label for="password"><i class="fas fa-lock"></i> New Password</label>
                        <i class="fas fa-lock input-icon"></i>
                        <input type="password" id="password" name="password" placeholder="Change your current password">
                    </div>
                    
                    <div class="form-group full-width">
                        <label><i class="fas fa-camera"></i> Profile Photo</label>
                        <div class="photo-upload">
                            <img src="<?php echo htmlspecialchars($path); ?>" alt="Current Profile Photo" class="current-photo" id="photo-preview">
                            <label for="photo" class="upload-btn">
                                <i class="fas fa-camera"></i> Change Photo
                            </label>
                            <input type="file" id="photo" name="photo" accept="image/*" onchange="previewPhoto(event)">
                        </div>
                    </div>
                    
                </div>
                
                <div class="btn-container">
                    <button type="submit" name="update_btn" class="btn">
                        <i class="fas fa-save"></i> Update Profile
                    </button>
                </div>
            </form>
            
        </div>
    </div>

    <script>
        function previewPhoto(event) {
            const reader = new FileReader();
            reader.onload = function() {
                const output = document.getElementById('photo-preview');
                output.src = reader.result;
            }
            if (event.target.files.length > 0) {
                 reader.readAsDataURL(event.target.files[0]);
            }
        }
        
        // Add interactive animations
        document.addEventListener('DOMContentLoaded', function() {
            const inputs = document.querySelectorAll('input[type="text"], input[type="email"], input[type="password"]'); 
            inputs.forEach(input => {
                const parent = input.closest('.form-group');
                if (parent) {
                    input.addEventListener('focus', function() {
                        if(window.innerWidth > 768) {
                            parent.style.transition = 'all 0.3s ease';
                            parent.style.transform = 'translateY(-2px)';
                            parent.style.boxShadow = '0 5px 15px rgba(0, 0, 0, 0.1)';
                        }
                    });
                    
                    input.addEventListener('blur', function() {
                        if(window.innerWidth > 768) {
                            parent.style.transform = 'translateY(0)';
                            parent.style.boxShadow = 'none';
                        }
                    });
                }
            });
        });
    </script>
</body>
</html>
<?php
// End output buffering and send content
if (ob_get_level() > 0) ob_end_flush();
?>