<?php
// teacher_home.php

// -------------------------------------------------------------------------
// CRITICAL FIX: Conditionally start the session to avoid the 'session_start(): Ignoring...' Notice.
// The session is already started in teacher_dashboard.php before this file is included.
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}
// -------------------------------------------------------------------------

ini_set('display_errors',1);
error_reporting(E_ALL);

// 1. Authorization Check
if(!isset($_SESSION['teacher_id'])){
    // The main dashboard should handle this, but it's good practice for included files too.
    echo "Not authorized"; 
    exit;
}

// Convert ID to integer for secure prepared statements (assuming 'id' is INT in DB)
$teacher_id = (int)$_SESSION['teacher_id'];
$teacher_name = $_SESSION['teacher_name'] ?? "Teacher"; // Use session name as fallback
$class_list = [];
$subject_list = [];

// 2. Database Configuration
$DB_CONFIG = [
    'teach_server' => "localhost",
    'teach_user' => "thevkin_schlteachdb", 
    'teach_pass' => "VK_AutSchlDb", 
    'teach_name' => "thevkin_schlteachdb",

    'core_server' => "localhost",
    'core_user' => "thevkin_schlcoredb", 
    'core_pass' => "VK_AutSchlDb", 
    'core_name' => "thevkin_schlcoredb"
];

// Establish connections
$teach_conn = @mysqli_connect($DB_CONFIG['teach_server'], $DB_CONFIG['teach_user'], $DB_CONFIG['teach_pass'], $DB_CONFIG['teach_name']);
$core_conn = @mysqli_connect($DB_CONFIG['core_server'], $DB_CONFIG['core_user'], $DB_CONFIG['core_pass'], $DB_CONFIG['core_name']);

if(!$teach_conn || !$core_conn) {
    if($teach_conn) mysqli_close($teach_conn);
    if($core_conn) mysqli_close($core_conn);
    // Log error instead of showing DB password!
    die("DB Error: Could not connect to one or both databases. Please check system logs.");
}


// --------------------------------------------------------
// A. Fetch Teacher Profile Details (Update $teacher_name)
// --------------------------------------------------------
// The teacher_id is assumed to be an integer (using 'i') for the teacher_profiles table
$stmt_profile = $teach_conn->prepare("SELECT first_name, last_name FROM teacher_profiles WHERE id = ?");
if ($stmt_profile) {
    $stmt_profile->bind_param("i", $teacher_id);
    $stmt_profile->execute();
    $result_profile = $stmt_profile->get_result();
    
    if ($profile = $result_profile->fetch_assoc()) {
        $teacher_name = htmlspecialchars($profile['first_name'] . ' ' . $profile['last_name']);
    }
    $stmt_profile->close();
}


// --------------------------------------------------------
// B. Fetch Class Teacher Assignments (Optimized Batch Fetch)
// --------------------------------------------------------
$class_ids = [];

// Query 1: Get Class IDs from Teacher DB
$stmt_ct = $teach_conn->prepare("SELECT class_id FROM teacher_class_assignments WHERE teacher_profile_id = ? AND is_class_teacher = 1");
if ($stmt_ct) {
    $stmt_ct->bind_param("i", $teacher_id);
    $stmt_ct->execute();
    $result_ct = $stmt_ct->get_result();
    
    while ($r = $result_ct->fetch_assoc()) {
        $class_ids[] = (int)$r['class_id'];
    }
    $stmt_ct->close();
}

// Query 2: Get Class Names from Core DB using IN clause
if (!empty($class_ids)) {
    $ids_string = implode(',', array_map('intval', $class_ids));
    // Using 'name' for the classes table as per schema
    $class_res = $core_conn->query("SELECT id, name FROM classes WHERE id IN ({$ids_string})");
    
    if ($class_res) {
        while ($c = $class_res->fetch_assoc()) {
            $class_list[] = $c;
        }
        $class_res->free();
    }
}


// --------------------------------------------------------
// C. Fetch Subject Assignments (Optimized Batch Fetch)
// --------------------------------------------------------
$subject_class_assignments = [];

// Query 1: Get Class ID and Subject ID pairs from Teacher DB
$stmt_st = $teach_conn->prepare("SELECT class_id, subject_id FROM teacher_subject_assignments WHERE teacher_profile_id = ?");
if ($stmt_st) {
    $stmt_st->bind_param("i", $teacher_id);
    $stmt_st->execute();
    $result_st = $stmt_st->get_result();
    
    while ($r = $result_st->fetch_assoc()) {
        $subject_class_assignments[] = ['class_id' => (int)$r['class_id'], 'subject_id' => (int)$r['subject_id']];
    }
    $stmt_st->close();
}

// Query 2: Get Class and Subject Names from Core DB (Batch fetch)
if (!empty($subject_class_assignments)) {
    $class_ids_st = array_unique(array_column($subject_class_assignments, 'class_id'));
    $subject_ids_st = array_unique(array_column($subject_class_assignments, 'subject_id'));
    
    // Fetch all required names
    $all_ids_classes = implode(',', array_map('intval', $class_ids_st));
    $all_ids_subjects = implode(',', array_map('intval', $subject_ids_st));
    
    $classes_map = [];
    if (!empty($all_ids_classes)) {
        // Using 'name' for the classes table as per schema
        $res = $core_conn->query("SELECT id, name FROM classes WHERE id IN ({$all_ids_classes})");
        if ($res) {
            while ($row = $res->fetch_assoc()) { $classes_map[$row['id']] = $row; }
            $res->free();
        }
    }
    
    $subjects_map = [];
    if (!empty($all_ids_subjects)) {
        // Using 'subject_name' and aliasing it to 'name'
        $res = $core_conn->query("SELECT id, subject_name AS name FROM subjects WHERE id IN ({$all_ids_subjects})");
        if ($res) {
            while ($row = $res->fetch_assoc()) { $subjects_map[$row['id']] = $row; }
            $res->free();
        }
    }

    // Combine assignment data with fetched names
    foreach ($subject_class_assignments as $assignment) {
        if (isset($subjects_map[$assignment['subject_id']]) && isset($classes_map[$assignment['class_id']])) {
            $subject_list[] = [
                'subject' => $subjects_map[$assignment['subject_id']],
                'class' => $classes_map[$assignment['class_id']]
            ];
        }
    }
}


// Close both database connections
mysqli_close($teach_conn);
if ($core_conn) mysqli_close($core_conn); // Added safety check
?>
<style>
/* Base Styles for Perfect UI (No changes needed here) */
body {
    font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, "Helvetica Neue", Arial, sans-serif;
    background-color: #f4f7f9;
}

.home-container {
    padding: 32px;
    max-width: 1200px; /* Limit width for desktop readability */
    margin: 0 auto; /* Center on desktop */
}

/* Welcome Banner */
.welcome-banner {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    padding: 40px;
    border-radius: 16px;
    margin-bottom: 32px;
    box-shadow: 0 10px 30px rgba(102, 126, 234, 0.4); /* Stronger shadow */
}

.welcome-banner h1 {
    font-size: 32px; /* Slightly larger on desktop */
    margin-bottom: 8px;
    font-weight: 700;
}

.welcome-banner p {
    font-size: 18px;
    opacity: 0.95;
    font-weight: 300;
}

/* Stats Grid */
.stats-grid {
    display: grid;
    grid-template-columns: repeat(3, 1fr); /* Default 3 columns */
    gap: 24px;
    margin-bottom: 32px;
}

.stat-card {
    background: white;
    padding: 24px;
    border-radius: 12px;
    border: 1px solid #e5e7eb;
    display: flex;
    align-items: center;
    gap: 16px;
    transition: all 0.3s ease;
}

.stat-card:hover {
    box-shadow: 0 8px 20px rgba(0, 0, 0, 0.08); /* Improved hover effect */
    transform: translateY(-4px);
}

.stat-icon {
    width: 60px; /* Slightly larger icon */
    height: 60px;
    border-radius: 12px;
    display: flex;
    align-items: center;
    justify-content: center;
    flex-shrink: 0;
}

/* Colors for Stat Icons */
.stat-icon.purple { background: linear-gradient(135deg, #a78bfa 0%, #7c3aed 100%); }
.stat-icon.blue { background: linear-gradient(135deg, #3b82f6 0%, #1d4ed8 100%); }
.stat-icon.green { background: linear-gradient(135deg, #10b981 0%, #047857 100%); }

.stat-icon svg {
    width: 30px;
    height: 30px;
    color: white;
}

.stat-info h3 {
    font-size: 36px;
    margin-bottom: 4px;
    color: #1f2937;
    font-weight: 600;
}

.stat-info p {
    font-size: 14px;
    color: #6b7280;
    margin: 0;
}

/* Section Cards */
.section-card {
    background: white;
    border-radius: 16px;
    padding: 28px;
    margin-bottom: 24px;
    border: 1px solid #e5e7eb;
    box-shadow: 0 2px 10px rgba(0, 0, 0, 0.05); /* Soft shadow for depth */
}

.section-header {
    display: flex;
    align-items: center;
    gap: 16px; /* Increased gap */
    margin-bottom: 20px;
    padding-bottom: 16px;
    border-bottom: 1px solid #e5e7eb; /* Thinner, softer divider */
}

.section-header h2 {
    font-size: 22px;
    color: #1f2937;
    font-weight: 600;
}

/* Assignment List */
.assignment-grid {
    display: flex;
    flex-direction: column;
    gap: 12px;
}

.assignment-item {
    background: #ffffff;
    padding: 18px 20px; /* Slightly more vertical padding */
    border-radius: 10px;
    border: 1px solid #f3f4f6;
    border-left: 5px solid #667eea; /* Prominent color strip */
    display: flex;
    align-items: center;
    justify-content: space-between;
    transition: all 0.2s ease;
}

.assignment-item:hover {
    background: #f9fafb;
    box-shadow: 0 2px 8px rgba(102, 126, 234, 0.1);
}

.assignment-info {
    display: flex;
    align-items: center;
    gap: 16px;
}

.assignment-details h4 {
    font-size: 16px;
    color: #1f2937;
    margin: 0 0 4px 0;
    font-weight: 500;
}

.assignment-details p {
    font-size: 13px;
    color: #6b7280;
    margin: 0;
}

.assignment-id {
    font-size: 13px;
    color: #4b5563;
    background: #f3f4f6;
    padding: 6px 14px;
    border-radius: 6px;
    font-weight: 500;
    flex-shrink: 0; /* Prevent ID from shrinking on desktop */
}

/* Empty State (MODIFIED FOR BIGGER ICON) */
.empty-state {
    text-align: center;
    padding: 60px 24px;
    background: #fcfcfc;
    border-radius: 10px;
    border: 1px dashed #e5e7eb; /* Added dashed border for better separation */
}

.empty-state h3 {
    font-size: 18px;
    color: #4b5563;
    margin-top: 16px;
}

.empty-state p {
    color: #9ca3af;
    margin-top: 8px;
}

.empty-icon {
    /* Style for the container of the large icon */
    width: 80px;
    height: 80px;
    margin: 0 auto 10px;
    display: flex;
    align-items: center;
    justify-content: center;
    border-radius: 50%;
    background-color: #eef2ff; /* Light background for the icon */
}

.empty-icon svg {
    /* Style for the large SVG icon */
    width: 40px;
    height: 40px;
    color: #6366f1; /* Primary blue/indigo color */
}

/* Mobile Responsiveness (max-width 768px for tablets/large phones) */
@media (max-width: 992px) {
    .stats-grid {
        grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); /* Allow 2 or 3 columns on tablet */
    }
}

@media (max-width: 768px) {
    .home-container {
        padding: 16px; /* Smaller padding on phones */
    }
    
    .welcome-banner {
        padding: 24px;
        margin-bottom: 24px;
    }
    
    .welcome-banner h1 {
        font-size: 26px; /* Optimized font size for mobile */
    }

    .welcome-banner p {
        font-size: 16px;
    }
    
    .stats-grid {
        grid-template-columns: 1fr; /* Stack stats vertically on phones */
        gap: 16px;
    }
    
    .stat-info h3 {
        font-size: 32px;
    }

    .section-card {
        padding: 20px;
    }

    /* Stack assignment items vertically on mobile for better flow */
    .assignment-item {
        flex-direction: column;
        align-items: flex-start;
        gap: 10px;
    }

    .assignment-info {
        width: 100%; /* Full width for details block */
    }

    .assignment-details h4 {
        font-size: 15px;
    }

    .assignment-id {
        margin-top: 8px;
        align-self: flex-end; /* Align the ID to the right/end */
    }
}
</style>

<div class="home-container">
    <div class="welcome-banner">
        <h1>Welcome back, <?php echo $teacher_name; ?>! 👋</h1>
        <p>Here's your teaching overview for today</p>
    </div>
    
    <div class="stats-grid">
        <div class="stat-card">
            <div class="stat-icon purple">
                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                    <path d="M3 9l9-7 9 7v11a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2z"/>
                    <polyline points="9 22 9 12 15 12 15 22"/>
                </svg>
            </div>
            <div class="stat-info">
                <h3><?php echo count($class_list); ?></h3>
                <p>Classes Managed</p>
            </div>
        </div>
        
        <div class="stat-card">
            <div class="stat-icon blue">
                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                    <path d="M2 3h6a4 4 0 0 1 4 4v14a3 3 0 0 0-3-3H2z"/>
                    <path d="M22 3h-6a4 4 0 0 0-4 4v14a3 3 0 0 1 3-3h7z"/>
                </svg>
            </div>
            <div class="stat-info">
                <h3><?php echo count($subject_list); ?></h3>
                <p>Subjects Taught</p>
            </div>
        </div>
        
        <div class="stat-card">
            <div class="stat-icon green">
                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                    <rect x="3" y="4" width="18" height="18" rx="2" ry="2"/>
                    <line x1="16" y1="2" x2="16" y2="6"/>
                    <line x1="8" y1="2" x2="8" y2="6"/>
                    <line x1="3" y1="10" x2="21" y2="10"/>
                </svg>
            </div>
            <div class="stat-info">
                <h3><?php echo date('d'); ?></h3>
                <p><?php echo date('F Y'); ?></p>
            </div>
        </div>
    </div>
    
    <div class="section-card">
        <div class="section-header">
            <div class="section-icon">
                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                    <path d="M3 9l9-7 9 7v11a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2z"/>
                    <polyline points="9 22 9 12 15 12 15 22"/>
                </svg>
            </div>
            <h2>Assigned as Class Teacher</h2>
        </div>
        
        <?php if(count($class_list)==0): ?>
        <div class="empty-state">
            <div class="empty-icon">
                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                    <path d="M3 9l9-7 9 7v11a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2z"/>
                    <polyline points="9 22 9 12 15 12 15 22"/>
                </svg>
            </div>
            <h3>No Class Assignments</h3>
            <p>You are not currently assigned as a Class Teacher for any section.</p>
        </div>
        <?php else: ?>
        <div class="assignment-grid">
            <?php foreach($class_list as $c): ?>
            <div class="assignment-item">
                <div class="assignment-info">
                    <div class="assignment-badge">
                        <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" width="18" height="18">
                            <path d="M3 9l9-7 9 7v11a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2z"/>
                        </svg>
                    </div>
                    <div class="assignment-details">
                        <h4><?php echo htmlspecialchars($c['name']); ?></h4>
                        <p>Class Teacher Role</p>
                    </div>
                </div>
                <div class="assignment-id">Class ID: <?php echo htmlspecialchars($c['id']); ?></div>
            </div>
            <?php endforeach; ?>
        </div>
        <?php endif; ?>
    </div>
    
    <div class="section-card">
        <div class="section-header">
            <div class="section-icon">
                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                    <path d="M2 3h6a4 4 0 0 1 4 4v14a3 3 0 0 0-3-3H2z"/>
                    <path d="M22 3h-6a4 4 0 0 0-4 4v14a3 3 0 0 1 3-3h7z"/>
                </svg>
            </div>
            <h2>Assigned as Subject Teacher</h2>
        </div>
        
        <?php if(count($subject_list)==0): ?>
        <div class="empty-state">
            <div class="empty-icon">
                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                    <path d="M2 3h6a4 4 0 0 1 4 4v14a3 3 0 0 0-3-3H2z"/>
                    <path d="M22 3h-6a4 4 0 0 0-4 4v14a3 3 0 0 1 3-3h7z"/>
                </svg>
            </div>
            <h3>No Subject Assignments</h3>
            <p>You do not have any subject teaching assignments for the current academic year.</p>
        </div>
        <?php else: ?>
        <div class="assignment-grid">
            <?php foreach($subject_list as $it): ?>
            <div class="assignment-item">
                <div class="assignment-info">
                    <div class="assignment-badge">
                        <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" width="18" height="18">
                            <path d="M2 3h6a4 4 0 0 1 4 4v14a3 3 0 0 0-3-3H2z"/>
                            <path d="M22 3h-6a4 4 0 0 0-4 4v14a3 3 0 0 1 3-3h7z"/>
                        </svg>
                    </div>
                    <div class="assignment-details">
                        <h4><?php echo htmlspecialchars($it['subject']['name']); ?></h4>
                        <p>Assigned to Class: <?php echo htmlspecialchars($it['class']['name']); ?></p>
                    </div>
                </div>
            </div>
            <?php endforeach; ?>
        </div>
        <?php endif; ?>
    </div>
</div>