<?php
// view_students.php (View/List Students for assigned classes)

// CRITICAL FIX: Conditionally start the session to avoid the "Ignoring session_start()" notice.
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

ini_set('display_errors',1);
error_reporting(E_ALL);

// --- Configuration and Initialization ---
// 1. Authorization Check
if(!isset($_SESSION['teacher_id'])){
    echo "Not authorized";
    exit;
}

$teacher_id = (int)$_SESSION['teacher_id'];
$selected_ay_id = null;
$admin_id = null;
$academic_years = [];

// Assignment and Student Data
$class_teacher_for = [];
$assigned_subject_details = [];
$available_class_ids = [];
$classes_map = [];
$students_data = [];
$selected_class_id = null;


// 2. Database Configuration
$DB_CONFIG = [
    'teach_server' => "localhost",
    'teach_user' => "thevkin_schlteachdb",
    'teach_pass' => "VK_AutSchlDb",
    'teach_name' => "thevkin_schlteachdb",

    'core_server' => "localhost",
    'core_user' => "thevkin_schlcoredb",
    'core_pass' => "VK_AutSchlDb",
    'core_name' => "thevkin_schlcoredb"
];

// Establish connections
$teach_conn = @mysqli_connect($DB_CONFIG['teach_server'], $DB_CONFIG['teach_user'], $DB_CONFIG['teach_pass'], $DB_CONFIG['teach_name']);
$core_conn = @mysqli_connect($DB_CONFIG['core_server'], $DB_CONFIG['core_user'], $DB_CONFIG['core_pass'], $DB_CONFIG['core_name']);

if(!$teach_conn || !$core_conn) {
    if($teach_conn) mysqli_close($teach_conn);
    if($core_conn) mysqli_close($core_conn);
    die("DB Error: Could not connect to one or both databases.");
}

// --------------------------------------------------------
// A. Determine School Admin ID (Scopes all data)
// --------------------------------------------------------
$stmt_admin = $teach_conn->prepare("SELECT created_by FROM teacher_profiles WHERE id = ?");
if ($stmt_admin && $teacher_id > 0) {
    $stmt_admin->bind_param("i", $teacher_id);
    $stmt_admin->execute();
    $result_admin = $stmt_admin->get_result();
    if ($r = $result_admin->fetch_assoc()) {
        $admin_id = $r['created_by'];
    }
    $stmt_admin->close();
}

if (!$admin_id) {
    mysqli_close($teach_conn);
    mysqli_close($core_conn);
    die("Error: Could not determine the School Admin ID for this teacher.");
}


// --------------------------------------------------------
// B. Fetch Academic Years (Scoped by Admin ID) & Determine Selected AY
// --------------------------------------------------------
$stmt_ay = $core_conn->prepare("SELECT id, Year AS name FROM academic_years WHERE created_by = ? ORDER BY id DESC");
if ($stmt_ay) {
    $stmt_ay->bind_param("s", $admin_id);
    $stmt_ay->execute();
    $ay_result = $stmt_ay->get_result();
    while ($row = $ay_result->fetch_assoc()) {
        $academic_years[] = $row;
    }
    $stmt_ay->close();
}

if (!empty($academic_years)) {
    $ay_ids = array_column($academic_years, 'id');
    $default_ay_id = $academic_years[0]['id'];

    $requested_id = (int)($_GET['academic_year_id'] ?? $default_ay_id);
    
    if (in_array($requested_id, $ay_ids)) {
        $selected_ay_id = $requested_id;
    } else {
        $selected_ay_id = $default_ay_id;
    }
    $selected_ay_id = (int)$selected_ay_id;
}


// --------------------------------------------------------
// C. Find Assigned Classes and Subjects (Combined Logic)
// --------------------------------------------------------
if ($selected_ay_id > 0) {
    
    $class_teacher_for_ids = [];
    $subject_ids_to_fetch = [];
    $raw_subject_assignments = [];
    
    $stmt_assignments = $teach_conn->prepare("
        SELECT
            TCA.class_id,
            TSA.subject_id,
            TCA.is_class_teacher
        FROM
            teacher_class_assignments TCA
        LEFT JOIN
            teacher_subject_assignments TSA ON TCA.class_id = TSA.class_id
            AND TCA.teacher_profile_id = TSA.teacher_profile_id
            AND TCA.academic_year_id = TSA.academic_year_id
        WHERE
            TCA.teacher_profile_id = ? AND TCA.academic_year_id = ?
        
        UNION
        
        SELECT
            TSA_only.class_id,
            TSA_only.subject_id,
            0 as is_class_teacher
        FROM
            teacher_subject_assignments TSA_only
        WHERE
            TSA_only.teacher_profile_id = ? AND TSA_only.academic_year_id = ?
    ");

    if ($stmt_assignments) {
        $stmt_assignments->bind_param("iiii",
            $teacher_id, $selected_ay_id, 
            $teacher_id, $selected_ay_id
        );
        $stmt_assignments->execute();
        $result_assignments = $stmt_assignments->get_result();
        
        while ($r = $result_assignments->fetch_assoc()) {
            $class_id = (int)$r['class_id'];
            $subject_id = $r['subject_id'] ? (int)$r['subject_id'] : null;

            if ($r['is_class_teacher'] == 1) {
                $class_teacher_for_ids[] = $class_id;
            }

            if ($subject_id !== null) {
                $raw_subject_assignments[] = ['class_id' => $class_id, 'subject_id' => $subject_id];
                $subject_ids_to_fetch[] = $subject_id;
            }
            
            $available_class_ids[] = $class_id;
        }
        $stmt_assignments->close();
    }
    
    $available_class_ids = array_unique($available_class_ids);
    $class_teacher_for_ids = array_unique($class_teacher_for_ids);
    $subject_ids_to_fetch = array_unique($subject_ids_to_fetch);

    $subjects_map = [];
    
    if (!empty($available_class_ids)) {
        $class_ids_string = implode(',', $available_class_ids);
        
        $stmt_classes = $core_conn->prepare("
            SELECT id, name
            FROM classes
            WHERE id IN ({$class_ids_string})
            AND academic_year_id = ?
            AND created_by = ?
        ");
        if ($stmt_classes) {
            $stmt_classes->bind_param("is", $selected_ay_id, $admin_id);
            $stmt_classes->execute();
            $res_classes = $stmt_classes->get_result();
            while ($row = $res_classes->fetch_assoc()) {
                $classes_map[$row['id']] = $row['name'];
            }
            $stmt_classes->close();
        }
        
        $available_class_ids = array_keys($classes_map);
    }
    
    if (!empty($subject_ids_to_fetch)) {
        $subject_ids_string = implode(',', $subject_ids_to_fetch);
        $stmt_subjects_map = $core_conn->prepare("
            SELECT id, subject_name
            FROM subjects
            WHERE id IN ({$subject_ids_string})
            AND academic_year_id = ?
            AND created_by = ?
        ");
        if ($stmt_subjects_map) {
            $stmt_subjects_map->bind_param("is", $selected_ay_id, $admin_id);
            $stmt_subjects_map->execute();
            $res_subjects = $stmt_subjects_map->get_result();
            while ($row = $res_subjects->fetch_assoc()) {
                $subjects_map[$row['id']] = $row['subject_name'];
            }
            $stmt_subjects_map->close();
        }
    }
    
    foreach ($class_teacher_for_ids as $cid) {
        if (isset($classes_map[$cid])) {
            $class_teacher_for[] = htmlspecialchars($classes_map[$cid]);
        }
    }

    $unique_subject_assignments = [];
    foreach ($raw_subject_assignments as $assignment) {
        $class_id = $assignment['class_id'];
        $subject_id = $assignment['subject_id'];
        
        if(isset($classes_map[$class_id]) && isset($subjects_map[$subject_id])){
            $class_name = $classes_map[$class_id];
            $subject_name = $subjects_map[$subject_id];
            
            if (!isset($unique_subject_assignments[$class_id])) {
                 $unique_subject_assignments[$class_id] = [
                    'class_name' => $class_name,
                    'subjects' => []
                ];
            }
            $unique_subject_assignments[$class_id]['subjects'][$subject_id] = htmlspecialchars($subject_name);
        }
    }
    
    foreach($unique_subject_assignments as $class_id => $detail) {
        $detail['subjects'] = array_values($detail['subjects']); 
        $assigned_subject_details[$class_id] = $detail;
    }
}


// --------------------------------------------------------
// D. Handle Class Filter and Determine Class(es) to Fetch Students For
// --------------------------------------------------------

if (isset($_GET['class_id']) && is_numeric($_GET['class_id'])) {
    $requested_class_id = (int)$_GET['class_id'];
    if (in_array($requested_class_id, $available_class_ids)) {
        $selected_class_id = $requested_class_id;
    }
}

$student_fetch_classes = [];

if ($selected_class_id !== null) {
    $student_fetch_classes = [$selected_class_id];
} else {
    $student_fetch_classes = $available_class_ids;
}


// --------------------------------------------------------
// E. Fetch Students (Scoped by selected class(es), AY, and Admin ID)
// --------------------------------------------------------

if (!empty($student_fetch_classes) && $selected_ay_id > 0) {
    $class_ids_string = implode(',', $student_fetch_classes);

    $sql_students = "
        SELECT
            id,
            class_id,
            name,
            roll_no,
            mobile_no,
            parent_name
        FROM
            students
        WHERE
            academic_year_id = ? AND class_id IN ({$class_ids_string}) AND created_by = ?
        ORDER BY
            class_id ASC, roll_no + 0 ASC
    ";
    
    $stmt_students = $core_conn->prepare($sql_students);
    
    if ($stmt_students) {
        $stmt_students->bind_param("is", $selected_ay_id, $admin_id);
        $stmt_students->execute();
        $result_students = $stmt_students->get_result();
        
        while ($row = $result_students->fetch_assoc()) {
            $students_data[] = $row;
        }
        $stmt_students->close();
    }
}

// Close database connections
mysqli_close($teach_conn);
mysqli_close($core_conn);
?>

<meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no">

<style>
/* Reset for aggressive mobile responsiveness */
html, body {
    /* Set margin/padding to 0 and hide overflow on the root elements */
    width: 100%;
    margin: 0;
    padding: 0;
    overflow-x: hidden; 
}

*, *::before, *::after {
    /* Critical box model fix */
    box-sizing: border-box; 
}


/* Base Styles */
body {
    font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, "Helvetica Neue", Arial, sans-serif;
    background-color: #f4f7f9;
}

.student-container {
    padding: 20px;
    max-width: 1200px;
    margin: 0 auto;
    /* FIX: Force max-width 100% and hide any overflow */
    width: 100%; 
    overflow-x: hidden; 
}

.header {
    background-color: #667eea;
    color: white;
    padding: 25px 30px;
    border-radius: 12px;
    margin-bottom: 20px;
    box-shadow: 0 4px 10px rgba(0, 0, 0, 0.1);
}

.header h1 {
    margin: 0;
    font-size: 28px;
    font-weight: 600;
}

.assignment-summary {
    margin-top: 15px;
    padding: 15px;
    background: #5a6edb;
    border-radius: 6px;
    font-size: 14px;
    line-height: 1.5;
}

.assignment-summary strong {
    font-weight: 700;
}

.assignment-summary ul {
    list-style: none;
    padding: 0;
    margin: 5px 0 0 0;
}

/* Filter Bar (Desktop default: horizontal) */
.filter-bar {
    background: white;
    padding: 15px 20px;
    border-radius: 8px;
    margin-bottom: 20px;
    display: flex;
    flex-wrap: wrap;
    align-items: center;
    gap: 20px;
    box-shadow: 0 2px 5px rgba(0, 0, 0, 0.05);
}

.filter-group {
    display: flex;
    align-items: center;
    gap: 10px;
}

.filter-bar label {
    font-weight: 500;
    color: #4b5563;
}

.filter-bar select {
    padding: 10px 15px;
    border: 1px solid #d1d5db;
    border-radius: 6px;
    font-size: 14px;
    background-color: #f9fafb;
    cursor: pointer;
    transition: border-color 0.2s;
    min-width: 180px;
}

/* Student Table Styles */
.student-table-wrapper {
    overflow-x: auto; /* Ensures horizontal scroll for the table on small screens */
    background: white;
    border-radius: 8px;
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.05);
    /* Set to 100% of the parent and make sure it behaves like a block */
    width: 100%;
    display: block; 
}

.student-table {
    width: 100%;
    border-collapse: collapse;
    min-width: 800px; /* Force min-width for desktop view/mobile scrollability */
}

.student-table th, .student-table td {
    padding: 15px 20px;
    text-align: left;
    border-bottom: 1px solid #e5e7eb;
}

.student-table th {
    background-color: #f3f4f6;
    color: #1f2937;
    font-weight: 600;
    text-transform: uppercase;
    font-size: 12px;
    white-space: nowrap;
}

.student-table tr:hover {
    background-color: #f9fafb;
}

.student-table td {
    font-size: 14px;
    color: #374151;
}

.class-badge {
    display: inline-block;
    padding: 4px 10px;
    background-color: #eef2ff;
    color: #4f46e5;
    border-radius: 4px;
    font-size: 12px;
    font-weight: 500;
    white-space: nowrap; 
}

/* Empty State */
.empty-state {
    text-align: center;
    padding: 60px 20px;
    background: white;
    border-radius: 8px;
    margin-top: 20px;
    border: 1px solid #e5e7eb;
}


/* -------------------------------------------------------- */
/* Mobile Specific Styles (Applied to screens <= 768px wide) */
/* -------------------------------------------------------- */
@media (max-width: 768px) {
    .student-container {
        padding: 10px;
        /* FIX 2: Use calc() to explicitly define width based on viewport width (100vw) 
                  minus the side padding. This is a powerful override. */
        width: calc(100vw - 20px); 
        max-width: 100%;
        margin-left: auto;
        margin-right: auto;
    }

    /* Stack filters vertically */
    .filter-bar {
        flex-direction: column;
        align-items: stretch;
        gap: 10px;
        padding: 10px;
        /* FIX: Ensure filter bar is also contained */
        width: 100%;
        max-width: 100%; 
    }

    .filter-group {
        width: 100%;
        flex-direction: column;
        align-items: flex-start;
        gap: 5px;
    }

    .filter-group select {
        width: 100%;
        min-width: unset;
    }
    
    /* Table adjustments for mobile */
    .student-table-wrapper {
        /* FIX 3: Ensure table wrapper is also forcefully constrained */
        width: 100%;
        max-width: 100%;
    }
    
    .student-table {
        min-width: 650px; 
    }

    .student-table th, .student-table td {
        padding: 8px 10px; 
        font-size: 12px;
    }

    /* Assign explicit (but still small) widths to columns to ensure scrollability */
    .student-table th:nth-child(1), .student-table td:nth-child(1) { /* Class */
        width: 15%; 
        min-width: 80px;
    }
    .student-table th:nth-child(2), .student-table td:nth-child(2) { /* Roll No. */
        width: 15%; 
        min-width: 80px;
    }
    .student-table th:nth-child(3), .student-table td:nth-child(3) { /* Student Name */
        width: 25%;
        min-width: 120px;
    }
    .student-table th:nth-child(4), .student-table td:nth-child(4) { /* Parent Name */
        width: 25%; 
        min-width: 120px;
    }
    .student-table th:nth-child(5), .student-table td:nth-child(5) { /* Phone Number */
        width: 20%; 
        min-width: 120px;
    }

    .class-badge {
        padding: 2px 5px;
        font-size: 10px;
    }
}
</style>

<div class="student-container">
    <div class="header">
        <h1>View Students</h1>
        <?php
            $ay_name = 'N/A';
            foreach ($academic_years as $ay_item) {
                if ($ay_item['id'] == $selected_ay_id) {
                    $ay_name = $ay_item['name'];
                    break;
                }
            }
        ?>
        <p>Viewing Students for Academic Year: **<?php echo htmlspecialchars($ay_name); ?>** (Admin ID: <?php echo htmlspecialchars($admin_id); ?>)</p>

        <div class="assignment-summary">
            <strong>Your Assignments in AY (<?php echo $selected_ay_id; ?>):</strong>
            <ul>
                <?php if (!empty($class_teacher_for)): ?>
                    <li>**Class Teacher For:** <?php echo implode(', ', $class_teacher_for); ?></li>
                <?php endif; ?>

                <?php if (!empty($assigned_subject_details)): ?>
                    <li>**Subject Teacher Assignments:**
                        <ul>
                            <?php foreach ($assigned_subject_details as $detail): ?>
                                <li>- Class **<?php echo htmlspecialchars($detail['class_name']); ?>**: (<?php echo implode(', ', $detail['subjects']); ?>)</li>
                            <?php endforeach; ?>
                        </ul>
                    </li>
                <?php endif; ?>

                <?php if (empty($class_teacher_for) && empty($assigned_subject_details)): ?>
                    <li>*No current assignments found in the system for this academic year (ID: **<?php echo $selected_ay_id; ?>**).*</li>
                <?php endif; ?>
            </ul>
        </div>
    </div>

    <div class="filter-bar">
        <div class="filter-group">
            <label for="academic_year_id">Academic Year:</label>
            <select id="academic_year_id" name="academic_year_id" onchange="
                var ay_id = this.value;
                var class_id = document.getElementById('class_filter_id').value;
                var base_url = 'teacher_dashboard.php?page=view_students&academic_year_id=' + ay_id;
                // Only pass class_id if it's set (not the default 'All Assigned Classes')
                if(class_id) {
                    base_url += '&class_id=' + class_id;
                } else {
                    // If switching AY, reset the class filter explicitly or implicitly
                    base_url += '&class_id=';
                }
                window.location.href = base_url;
            ">
                <?php foreach ($academic_years as $ay): ?>
                    <option
                        value="<?php echo htmlspecialchars($ay['id']); ?>"
                        <?php if ($ay['id'] == $selected_ay_id) echo 'selected'; ?>
                    >
                        <?php echo htmlspecialchars($ay['name']); ?>
                    </option>
                <?php endforeach; ?>
            </select>
        </div>

        <div class="filter-group">
            <label for="class_filter_id">Filter by Class:</label>
            <select id="class_filter_id" name="class_id" onchange="
                var ay_id = document.getElementById('academic_year_id').value;
                var class_id = this.value;
                window.location.href = 'teacher_dashboard.php?page=view_students&academic_year_id=' + ay_id + '&class_id=' + class_id;
            ">
                <option value="">-- All Assigned Classes --</option>
                <?php
                // Sort classes by name for better readability in the dropdown
                asort($classes_map);
                foreach ($classes_map as $cid => $cname):
                    // Only display classes the teacher is truly assigned to
                    if (in_array($cid, $available_class_ids)):
                ?>
                    <option
                        value="<?php echo htmlspecialchars($cid); ?>"
                        <?php if ($cid == $selected_class_id) echo 'selected'; ?>
                    >
                        <?php echo htmlspecialchars($cname); ?>
                    </option>
                <?php
                    endif;
                endforeach;
                ?>
            </select>
        </div>
    </div>

    <?php if ($selected_ay_id === null || empty($academic_years)): ?>
        <div class="empty-state">
            <h3>No Academic Years Found</h3>
            <p>The academic year list is empty or none exist for your school admin (ID: <?php echo htmlspecialchars($admin_id); ?>).</p>
        </div>
    <?php elseif (empty($available_class_ids)): ?>
        <div class="empty-state">
            <h3>No Class Assignments Found</h3>
            <p>You have no subject or class teacher assignments for the selected academic year (ID: **<?php echo $selected_ay_id; ?>**).</p>
        </div>
    <?php elseif (empty($students_data)): ?>
        <div class="empty-state">
            <h3>No Students Found</h3>
            <p>There are no students registered in your assigned class(es) (<?php echo $selected_class_id ? $classes_map[$selected_class_id] : 'All Assigned Classes'; ?>) for the selected academic year.</p>
        </div>
    <?php else: ?>
        <div class="student-table-wrapper">
            <table class="student-table">
                <thead>
                    <tr>
                        <th>Class</th>
                        <th>Roll No.</th>
                        <th>Student Name</th>
                        <th>Parent Name</th>
                        <th>Phone Number</th>
                    </tr>
                </thead>
                <tbody>
                    <?php
                    $current_class_id = null;
                    foreach ($students_data as $student):
                        if ($student['class_id'] !== $current_class_id && $selected_class_id === null):
                            $current_class_id = $student['class_id'];
                        endif;
                    ?>
                    <tr>
                        <td><span class="class-badge"><?php echo htmlspecialchars($classes_map[$student['class_id']] ?? 'Unknown'); ?></span></td>
                        <td><?php echo htmlspecialchars($student['roll_no']); ?></td>
                        <td><?php echo htmlspecialchars($student['name']); ?></td>
                        <td><?php echo htmlspecialchars($student['parent_name'] ?? 'N/A'); ?></td>
                        <td><?php echo htmlspecialchars($student['mobile_no'] ?? 'N/A'); ?></td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    <?php endif; ?>
</div>