<?php
// report_card_generator.php (For Class Teachers)

// Start session
session_start();

// Enable error reporting only for debugging (turn off in production)
error_reporting(E_ALL);
ini_set('display_errors', 1);

// ======================================================================
// 1. CONFIGURATION AND INITIAL SETUP
// ======================================================================

// --- Database Connection Details ---
define('DB_HOST', 'localhost');
define('DB_PASS', 'VK_AutSchlDb');

// 1. Teacher Database
define('DB_SCHLTEACH', 'thevkin_schlteachdb');
define('DB_USER_TEACH', 'thevkin_schlteachdb');

// 2. Core Database (Where reports will be saved)
define('DB_SCHLCORE', 'thevkin_schlcoredb');
define('DB_USER_CORE', 'thevkin_schlcoredb');

// 3. Auth/School Stamp Database
define('DB_SCHLAUT', 'thevkin_schlautdb');
define('DB_USER_AUT', 'thevkin_schlautdb');

// --- Paths and URLs ---
define('BASE_URL', 'https://teachers.thevk.in/');
define('BASE_IMAGE_URL', 'https://principals.thevk.in/');

// !!! CRITICAL: UPDATE THIS PATH TO YOUR ABSOLUTE SERVER PATH !!!
define('REPORT_STORAGE_DIR', '/home/thevkin/teachers.thevk.in/reports/pdfs/');

// --- Session Variables (Assumed from Login) ---
$teacher_id = $_SESSION['teacher_id'] ?? null;
$academic_year_id = $_SESSION['academic_year_id'] ?? 1;

// --- State Variables ---
$class_id = $_POST['class_id'] ?? null;
$exam_id = $_POST['exam_id'] ?? null;
$start_date = $_POST['attendance_start_date'] ?? null;
$end_date = $_POST['attendance_end_date'] ?? null;

$error = '';
$message = '';
$is_report_ready = false;
$report_data = [];
$students_in_class = [];
$generated_pdfs = [];

$conn_teach = false;
$conn_core = false;
$conn_aut = false;
$school_details = [];

// ======================================================================
// 1.1. CRITICAL DIRECTORY CHECK AND CREATION
// ======================================================================

if (!$teacher_id) {
    die("Access Denied: Please log in as a teacher. (Missing \$_SESSION['teacher_id'])");
}

// Attempt to create the directory structure
if (!is_dir(REPORT_STORAGE_DIR)) {
    if (!mkdir(REPORT_STORAGE_DIR, 0775, true)) {
        // Set a warning, but don't halt.
        $error = "WARNING: Could not create the reports directory. (Check permissions).<br>";
    }
}

// ======================================================================
// 2. DATABASE CONNECTION FUNCTION
// ======================================================================

function get_db_connection($db_name) {
    global $error;
    $db_user = '';
    
    switch ($db_name) {
        case DB_SCHLTEACH:
            $db_user = DB_USER_TEACH;
            break;
        case DB_SCHLCORE:
            $db_user = DB_USER_CORE;
            break;
        case DB_SCHLAUT:
            $db_user = DB_USER_AUT;
            break;
        default:
            $error .= "Configuration Error: Unknown database requested: $db_name<br>";
            return false;
    }
    
    $conn = new mysqli(DB_HOST, $db_user, DB_PASS, $db_name);
    if ($conn->connect_error) {
        $error .= "Database connection failed for **$db_name**: " . $conn->connect_error . "<br>";
        return false;
    }
    return $conn;
}

// ======================================================================
// 3. INITIAL DATA FETCH & SCHOOL DETAILS
// ======================================================================

// Connections are initiated here, used throughout the main script logic.
$conn_teach = get_db_connection(DB_SCHLTEACH);
$conn_core = get_db_connection(DB_SCHLCORE);

// 3.0. Fetch School Details
$conn_aut = get_db_connection(DB_SCHLAUT);

if ($conn_aut) {
    $sql_school = "SELECT school_name, stamp_image_path, signature_image_path FROM school_stamps LIMIT 1";
    $result_school = $conn_aut->query($sql_school);
    if ($result_school && $result_school->num_rows > 0) {
        $school_details = $result_school->fetch_assoc();
    }
    // Crucial: Close the connection immediately after use
    $conn_aut->close();
}

$available_exams = [];
$class_teacher_data = null;
$class_name = 'N/A';
$class_teacher_class_id = null;

if ($conn_teach && $conn_core) {
    
    // 3.1. Identify the Class Teacher's assigned class
    $sql_class = "
        SELECT class_id, is_class_teacher
        FROM teacher_class_assignments
        WHERE teacher_profile_id = ? AND academic_year_id = ? AND is_class_teacher = 1
    ";
    $stmt_class = $conn_teach->prepare($sql_class);
    $stmt_class->bind_param("ii", $teacher_id, $academic_year_id);
    $stmt_class->execute();
    $result_class = $stmt_class->get_result();
    
    if ($result_class->num_rows > 0) {
        $class_teacher_data = $result_class->fetch_assoc();
        $class_teacher_class_id = $class_teacher_data['class_id'];
    } else {
        $error .= "You are not assigned as a Class Teacher for any class this academic year.<br>";
        goto end_initial_fetch;
    }
    $stmt_class->close();
    $result_class->free();
    
    // 3.2. Fetch all available exams for the year
    $sql_exams = "SELECT id, name FROM exams WHERE academic_year_id = ? ORDER BY id DESC";
    $stmt_exams = $conn_core->prepare($sql_exams);
    $stmt_exams->bind_param("i", $academic_year_id);
    $stmt_exams->execute();
    $result_exams = $stmt_exams->get_result();
    while ($row = $result_exams->fetch_assoc()) {
        $available_exams[$row['id']] = $row['name'];
    }
    $stmt_exams->close();
    $result_exams->free();
    
    // 3.3. Fetch the Class Name and Students
    if ($class_teacher_class_id) {
        $sql_class_info = "SELECT name FROM classes WHERE id = ?";
        $stmt_class_info = $conn_core->prepare($sql_class_info);
        $stmt_class_info->bind_param("i", $class_teacher_class_id);
        $stmt_class_info->execute();
        $result_class_info = $stmt_class_info->get_result();
        $class_name = $result_class_info->fetch_assoc()['name'] ?? 'N/A';
        $stmt_class_info->close();
        $result_class_info->free();
        
        $sql_students = "SELECT id, name, roll_no FROM students WHERE class_id = ? AND academic_year_id = ? ORDER BY roll_no ASC";
        $stmt_students = $conn_core->prepare($sql_students);
        $stmt_students->bind_param("ii", $class_teacher_class_id, $academic_year_id);
        $stmt_students->execute();
        $result_students = $stmt_students->get_result();
        while ($row = $result_students->fetch_assoc()) {
            $students_in_class[] = $row;
        }
        $stmt_students->close();
        $result_students->free();
    }
}

end_initial_fetch:

// ======================================================================
// 4. PDF HANDLING FUNCTIONS (Database Logging Only)
// ======================================================================

/**
 * Saves the report record to the database with a mock PDF path.
 */
function save_report_record($conn_core, $data, $teacher_id) {
    global $academic_year_id, $message, $error;
    
    // --- 1. Define File Paths (For DB Log/Link only) ---
    $class_name_clean = str_replace(' ', '_', $data['class_name']);
    $exam_name_clean = str_replace(' ', '_', $data['exam_name']);
    $pdf_filename = "report_card_{$academic_year_id}_{$class_name_clean}_{$data['roll_no']}_{$exam_name_clean}.pdf";
    $db_path = 'reports/pdfs/' . $pdf_filename;
    
    // --- 2. Log Record to Database ---
    $sql_insert = "
        INSERT INTO generated_report_cards
        (academic_year_id, exam_id, class_id, student_id, roll_no, pdf_path, created_by)
        VALUES (?, ?, ?, ?, ?, ?, ?)
        ON DUPLICATE KEY UPDATE pdf_path=?, created_by=?, created_at=CURRENT_TIMESTAMP
    ";
    
    $stmt = $conn_core->prepare($sql_insert);
    
    if (!$stmt) {
        $error .= "Database prepare failed: " . $conn_core->error . "<br>";
        return false;
    }
    
    $stmt->bind_param(
        "iiiisssis",
        $academic_year_id,
        $data['exam_id'],
        $data['details']['class_id'],
        $data['id'],
        $data['roll_no'],
        $db_path,
        $teacher_id,
        $db_path,
        $teacher_id
    );
    
    if (!$stmt->execute()) {
        $error .= "Database save failed for {$data['name']}: " . $stmt->error . "<br>";
        $stmt->close();
        return false;
    }
    $stmt->close();
    
    $message .= "✅ Report record logged successfully for **{$data['name']}**.<br>";
    
    return true;
}

/**
 * Fetches existing PDF paths for a class/exam combination.
 */
function fetch_existing_pdfs($conn_core, $class_id, $exam_id, $academic_year_id) {
    $pdfs = [];
    $sql = "
        SELECT student_id, pdf_path, created_at, created_by
        FROM generated_report_cards
        WHERE class_id = ? AND exam_id = ? AND academic_year_id = ?
    ";
    $stmt = $conn_core->prepare($sql);
    if (!$stmt) return $pdfs;
    
    $stmt->bind_param("iii", $class_id, $exam_id, $academic_year_id);
    $stmt->execute();
    $result = $stmt->get_result();
    
    while ($row = $result->fetch_assoc()) {
        $pdfs[$row['student_id']] = $row;
    }
    
    $result->free();
    $stmt->close();
    return $pdfs;
}

// ======================================================================
// 5. REPORT GENERATION LOGIC
// ======================================================================

if ($conn_teach && $conn_core && $_SERVER['REQUEST_METHOD'] === 'POST') {
    
    $exam_id = filter_var($_POST['exam_id'], FILTER_VALIDATE_INT);
    $class_id = $class_teacher_class_id;
    $start_date = $_POST['attendance_start_date'];
    $end_date = $_POST['attendance_end_date'];
    $save_mode = isset($_POST['save_pdfs']);
    $regenerate_student_id = filter_var($_POST['regenerate_student_id'] ?? null, FILTER_VALIDATE_INT);
    
    if (!$exam_id || empty($students_in_class)) {
        $error .= "Please select a valid exam and ensure students are enrolled in your assigned class.<br>";
        goto end_report_generation;
    }
    
    if ($save_mode && !$regenerate_student_id) {
        $message = "Attempting to log report records to the database...<br>";
    }
    
    // 5.1. Define Grading Rules
    $grade_rules = [
        ['min' => 90, 'grade' => 'A+'],
        ['min' => 80, 'grade' => 'A'],
        ['min' => 70, 'grade' => 'B+'],
        ['min' => 60, 'grade' => 'B'],
        ['min' => 50, 'grade' => 'C'],
        ['min' => 40, 'grade' => 'D'],
        ['min' => 0, 'grade' => 'F'],
    ];
    
    // 5.2. Get all required subjects for this class
    $required_subjects = [];
    $sql_subjects = "SELECT id, subject_name FROM subjects WHERE class_id = ? AND academic_year_id = ?";
    $stmt_subjects = $conn_core->prepare($sql_subjects);
    $stmt_subjects->bind_param("ii", $class_id, $academic_year_id);
    $stmt_subjects->execute();
    $result_subjects = $stmt_subjects->get_result();
    while ($row = $result_subjects->fetch_assoc()) {
        $required_subjects[$row['id']] = $row['subject_name'];
    }
    $stmt_subjects->close();
    $result_subjects->free();
    
    if (empty($required_subjects)) {
        $error .= "No subjects are defined for your class: " . $class_name . "<br>";
        goto end_report_generation;
    }
    $subject_count = count($required_subjects);
    
    // 5.3. Fetch all marks for the selected class and exam
    $sql_marks = "
        SELECT student_id, subject_id, gained_marks, total_marks, passing_marks
        FROM marks
        WHERE class_id = ? AND exam_id = ? AND academic_year_id = ?
    ";
    $stmt_marks = $conn_core->prepare($sql_marks);
    $stmt_marks->bind_param("iii", $class_id, $exam_id, $academic_year_id);
    $stmt_marks->execute();
    $result_marks = $stmt_marks->get_result();
    
    $marks_data = [];
    while ($row = $result_marks->fetch_assoc()) {
        $marks_data[$row['student_id']][$row['subject_id']] = $row;
    }
    $stmt_marks->close();
    $result_marks->free();
    
    // --- Data Processing and Report Generation ---
    $is_all_data_complete = true;
    $students_to_process = $students_in_class;
    
    // If regenerating a single student, filter the list
    if ($regenerate_student_id) {
        $students_to_process = array_filter($students_in_class, function($s) use ($regenerate_student_id) {
            return $s['id'] == $regenerate_student_id;
        });
        
        $save_mode = true;
        $message = "Attempting to re-log report record for student ID: {$regenerate_student_id}...<br>";
    }
    
    $total_students_processed = 0;
    
    foreach ($students_to_process as $student) {
        $student_id = $student['id'];
        $student_name = $student['name'];
        $roll_no = $student['roll_no'];
        
        $student_report = [
            'subjects' => [],
            'sum_gained' => 0,
            'sum_total' => 0,
            'percentage' => 0,
            'overall_grade' => 'N/A',
            'attendance_percent' => 'N/A',
            'class_id' => $class_id
        ];
        
        // 5.4. Check for Mark Completion
        if (!isset($marks_data[$student_id]) || count($marks_data[$student_id]) < $subject_count) {
            $is_all_data_complete = false;
            if ($save_mode) {
                $error .= "Marks for student **{$student_name}** are incomplete or missing for this exam. Cannot proceed with saving.<br>";
                continue;
            } else {
                $error .= "Marks for student **{$student_name}** are incomplete or missing. (Previewing anyway)<br>";
            }
        }
        
        // Calculate Subject Totals and Marks
        $student_pass = true;
        foreach ($required_subjects as $sub_id => $sub_name) {
            $mark_row = $marks_data[$student_id][$sub_id] ?? ['total_marks' => 0, 'gained_marks' => 0, 'passing_marks' => 0];
            
            if ($mark_row['gained_marks'] < $mark_row['passing_marks']) {
                $student_pass = false;
            }
            
            $student_report['subjects'][] = [
                'name' => $sub_name,
                'total_marks' => $mark_row['total_marks'],
                'gained_marks' => $mark_row['gained_marks'],
                'grade' => get_grade($mark_row['gained_marks'], $mark_row['total_marks'], $grade_rules),
            ];
            
            $student_report['sum_gained'] += $mark_row['gained_marks'];
            $student_report['sum_total'] += $mark_row['total_marks'];
        }
        
        // Calculate Percentage and Overall Grade
        if ($student_report['sum_total'] > 0) {
            $percentage = round(($student_report['sum_gained'] / $student_report['sum_total']) * 100, 2);
            $student_report['percentage'] = $percentage;
            
            if (!$student_pass) {
                $student_report['overall_grade'] = 'F (Failed)';
            } else {
                $student_report['overall_grade'] = get_overall_grade($percentage, $grade_rules);
            }
        }
        
        // Calculate Attendance
        $student_report['attendance_percent'] = calculate_attendance($conn_core, $student_id, $class_id, $start_date, $end_date, $students_in_class);
        
        $report_data[$student_id] = [
            'id' => $student_id,
            'name' => $student_name,
            'roll_no' => $roll_no,
            'class_name' => $class_name,
            'exam_name' => $available_exams[$exam_id] ?? 'N/A',
            'exam_id' => $exam_id,
            'details' => $student_report
        ];
        
        // 5.5. PDF Saving Step (Only executes if Save button was pressed)
        if ($save_mode) {
            if (save_report_record($conn_core, $report_data[$student_id], $teacher_id)) {
                $total_students_processed++;
            }
        }
    }
    
    // Final user-friendly message after batch processing
    if ($save_mode && $regenerate_student_id) {
        if ($total_students_processed > 0) {
            $message = "✅ Report record for Roll No. **{$roll_no}** logged successfully! Click **'View Report'** to check the updated data.";
        } else {
            $message = "❌ Error: Could not re-log record for Roll No. **{$roll_no}**. Check marks or error log.";
        }
        
        $is_report_ready = false;
    } elseif ($save_mode && empty($error)) {
        $message = "✅ **Report records logged successfully!** Records for **{$total_students_processed}** students are ready. **NOW:** Click the **'Print/Download PDF (Browser)'** button to save the file(s) to your computer.";
    } elseif ($save_mode && !empty($error)) {
        $message = "❌ **Error during batch save.** Some students may have been skipped. Check the error details above.";
    }
    
    if (!$save_mode) {
        $is_report_ready = true;
        if (empty($error)) {
            $message = "Preview generated. Click **'Log Record'** to save the records, then use **'Print/Download'**.";
        }
    }
    
    // 5.6. Fetch existing PDFs for display
    if ($class_id && $exam_id) {
        $generated_pdfs = fetch_existing_pdfs($conn_core, $class_id, $exam_id, $academic_year_id);
    }
}

end_report_generation:

// ======================================================================
// 6. HELPER FUNCTIONS
// ======================================================================

function get_grade($gained, $total, $rules) {
    if ($total == 0 || $gained === null) return 'N/A';
    $percentage = ($gained / $total) * 100;
    
    foreach ($rules as $rule) {
        if ($percentage >= $rule['min']) {
            return $rule['grade'];
        }
    }
    return 'F';
}

function get_overall_grade($percentage, $rules) {
    foreach ($rules as $rule) {
        if ($percentage >= $rule['min']) {
            return $rule['grade'];
        }
    }
    return 'D';
}

function calculate_attendance($conn, $student_id, $class_id, $start_date, $end_date, $all_students) {
    if (empty($start_date) || empty($end_date)) {
        return '0.00%';
    }
    
    $roll_no_str = null;
    foreach($all_students as $s) {
        if ($s['id'] == $student_id) {
            $roll_no_str = $s['roll_no'];
            break;
        }
    }
    if (!$roll_no_str) return 'N/A';
    
    $sql_attn = "
        SELECT status
        FROM attendance
        WHERE student_rollno = ? AND class_id = ?
        AND attendance_date BETWEEN ? AND ?
    ";
    
    $stmt_attn = $conn->prepare($sql_attn);
    $stmt_attn->bind_param("isss", $roll_no_str, $class_id, $start_date, $end_date);
    
    $stmt_attn->execute();
    $result = $stmt_attn->get_result();
    
    $total_days = $result->num_rows;
    $present_days = 0;
    
    while ($row = $result->fetch_assoc()) {
        if (strtoupper($row['status']) == 'P' || strtoupper($row['status']) == 'PRESENT') {
            $present_days++;
        }
    }
    $result->free();
    $stmt_attn->close();
    
    if ($total_days == 0) {
        return '0.00%';
    }
    
    return round(($present_days / $total_days) * 100, 2) . '%';
}

/**
 * Generates the HTML structure for a single report card.
 */
function generate_report_card_html($data, $school_details) {
    global $BASE_IMAGE_URL; // Using global for BASE_IMAGE_URL
    
    $logo_path = !empty($school_details['stamp_image_path']) ? BASE_IMAGE_URL . $school_details['stamp_image_path'] : '';
    $sig_path = !empty($school_details['signature_image_path']) ? BASE_IMAGE_URL . $school_details['signature_image_path'] : '';
    $school_name = htmlspecialchars($school_details['school_name'] ?? 'School Name Missing');
    
    $html = '<div class="report-card">';
    
    // Header with Logo and School Info - MODIFIED for centered logo
    $html .= '  <div class="report-header">';
    if ($logo_path) {
        $html .= '      <img src="' . $logo_path . '" alt="School Logo" class="school-logo">';
    }
    $html .= '      <div class="school-info">';
    $html .= '          <h3>' . $school_name . '</h3>';
    $html .= '          <h4>REPORT CARD - ' . strtoupper(htmlspecialchars($data['exam_name'])) . '</h4>';
    $html .= '      </div>';
    $html .= '  </div>';
    
    // Horizontal separator
    $html .= '  <div class="header-separator"></div>';
    
    // Student Details Box
    $html .= '  <div class="report-details">';
    $html .= '      <div class="detail-item"><strong>Student Name:</strong> ' . htmlspecialchars($data['name']) . '</div>';
    $html .= '      <div class="detail-item"><strong>Date of Report:</strong> ' . date('d M, Y') . '</div>';
    $html .= '      <div class="detail-item"><strong>Class:</strong> ' . htmlspecialchars($data['class_name']) . '</div>';
    $html .= '      <div class="detail-item"><strong>Roll No:</strong> ' . htmlspecialchars($data['roll_no']) . '</div>';
    $html .= '      <div class="detail-item"><strong>Exam Name:</strong> ' . htmlspecialchars($data['exam_name']) . '</div>';
    $html .= '      <div class="detail-item"><strong>Attendance (%):</strong> ' . htmlspecialchars($data['details']['attendance_percent']) . '</div>';
    $html .= '  </div>';
    
    // Marks Table
    $html .= '  <table class="report-table">';
    $html .= '      <thead>';
    $html .= '          <tr>';
    $html .= '              <th class="subject-col">Name of Subjects</th>';
    $html .= '              <th>Total Marks (Max)</th>';
    $html .= '              <th>Gained Marks</th>';
    $html .= '              <th>Grade</th>';
    $html .= '          </tr>';
    $html .= '      </thead>';
    $html .= '      <tbody>';
    
    foreach ($data['details']['subjects'] as $subject) {
        $html .= '<tr>';
        $html .= '  <td class="subject-name">' . htmlspecialchars($subject['name']) . '</td>';
        $html .= '  <td>' . htmlspecialchars($subject['total_marks']) . '</td>';
        $html .= '  <td>' . htmlspecialchars($subject['gained_marks']) . '</td>';
        $html .= '  <td class="grade-cell">' . htmlspecialchars($subject['grade']) . '</td>';
        $html .= '</tr>';
    }
    
    // Total Row
    $html .= '          <tr class="total-row">';
    $html .= '              <td class="subject-name">TOTAL</td>';
    $html .= '              <td>' . htmlspecialchars($data['details']['sum_total']) . '</td>';
    $html .= '              <td>' . htmlspecialchars($data['details']['sum_gained']) . '</td>';
    $html .= '              <td class="grade-cell">' . htmlspecialchars($data['details']['overall_grade']) . '</td>';
    $html .= '          </tr>';
    
    // Percentage Row
    $html .= '          <tr class="percentage-row-table">';
    $html .= '              <td colspan="4" class="percentage-cell"><strong>**Overall Percentage:**</strong> ' . htmlspecialchars($data['details']['percentage']) . '%</td>';
    $html .= '          </tr>';
    
    $html .= '      </tbody>';
    $html .= '  </table>';
    
    // Signature Section
    $html .= '  <div class="signature-section">';
    $html .= '      <div class="signature-box parent-signature">';
    $html .= '          <div class="signature-line"></div>';
    $html .= '          <div class="signature-label">Parent Signature</div>';
    $html .= '      </div>';
    $html .= '      <div class="signature-box principal-signature">';
    if ($sig_path) {
        $html .= '          <img src="' . $sig_path . '" alt="Principal Signature" class="signature-img">';
    }
    $html .= '          <div class="signature-line"></div>';
    $html .= '          <div class="signature-label">Principal Signature</div>';
    $html .= '      </div>';
    $html .= '  </div>';
    
    $html .= '</div>';
    return $html;
}

// ======================================================================
// 7. SINGLE REPORT VIEW LOGIC
// ======================================================================

if (isset($_GET['view_report']) && isset($_GET['student_id']) && isset($_GET['exam_id'])) {
    
    $view_student_id = filter_var($_GET['student_id'], FILTER_VALIDATE_INT);
    $view_exam_id = filter_var($_GET['exam_id'], FILTER_VALIDATE_INT);
    
    $conn_core_view = get_db_connection(DB_SCHLCORE);
    $conn_teach_view = get_db_connection(DB_SCHLTEACH);
    $conn_aut_view = get_db_connection(DB_SCHLAUT);
    
    $view_school_details = [];
    if ($conn_aut_view) {
        $sql_school = "SELECT school_name, stamp_image_path, signature_image_path FROM school_stamps LIMIT 1";
        $result_school = $conn_aut_view->query($sql_school);
        if ($result_school && $result_school->num_rows > 0) {
            $view_school_details = $result_school->fetch_assoc();
        }
        $conn_aut_view->close();
    }
    
    if ($conn_core_view && $view_student_id && $view_exam_id) {
        
        $sql_student_info = "SELECT id, name, roll_no, class_id FROM students WHERE id = ? AND academic_year_id = ?";
        $stmt_info = $conn_core_view->prepare($sql_student_info);
        $stmt_info->bind_param("ii", $view_student_id, $academic_year_id);
        $stmt_info->execute();
        $result_info = $stmt_info->get_result();
        
        $student_info = $result_info->fetch_assoc();
        $result_info->free();
        $stmt_info->close();
        
        if ($student_info) {
            $class_teacher_class_id = $student_info['class_id'];
            $students_in_class_view = [$student_info];
            
            $report_data_view = [];
            
            // Fetch Class Name
            $sql_class_name = "SELECT name FROM classes WHERE id = ?";
            $stmt_cn = $conn_core_view->prepare($sql_class_name);
            $stmt_cn->bind_param("i", $class_teacher_class_id);
            $stmt_cn->execute();
            $result_cn = $stmt_cn->get_result();
            $class_name = $result_cn->fetch_assoc()['name'] ?? 'N/A';
            $result_cn->free();
            $stmt_cn->close();
            
            // Fetch Exam Name
            $sql_exam_name = "SELECT name FROM exams WHERE id = ?";
            $stmt_en = $conn_core_view->prepare($sql_exam_name);
            $stmt_en->bind_param("i", $view_exam_id);
            $stmt_en->execute();
            $result_en = $stmt_en->get_result();
            $exam_name = $result_en->fetch_assoc()['name'] ?? 'N/A';
            $result_en->free();
            $stmt_en->close();
            
            // Fetch subjects
            $required_subjects = [];
            $sql_subjects = "SELECT id, subject_name FROM subjects WHERE class_id = ? AND academic_year_id = ?";
            $stmt_subjects = $conn_core_view->prepare($sql_subjects);
            $stmt_subjects->bind_param("ii", $class_teacher_class_id, $academic_year_id);
            $stmt_subjects->execute();
            $result_subjects = $stmt_subjects->get_result();
            while ($row = $result_subjects->fetch_assoc()) {
                $required_subjects[$row['id']] = $row['subject_name'];
            }
            $result_subjects->free();
            $stmt_subjects->close();
            
            $subject_count = count($required_subjects);
            if ($subject_count == 0) {
                $error .= "No subjects defined for this class.";
                goto end_single_view;
            }
            
            // Fetch marks
            $sql_marks = "
                SELECT subject_id, gained_marks, total_marks, passing_marks
                FROM marks
                WHERE student_id = ? AND exam_id = ? AND academic_year_id = ?
            ";
            $stmt_marks = $conn_core_view->prepare($sql_marks);
            $stmt_marks->bind_param("iii", $view_student_id, $view_exam_id, $academic_year_id);
            $stmt_marks->execute();
            $result_marks = $stmt_marks->get_result();
            
            $marks_data = [];
            while ($row = $result_marks->fetch_assoc()) {
                $marks_data[$row['subject_id']] = $row;
            }
            $result_marks->free();
            $stmt_marks->close();
            
            // Data Processing
            $grade_rules = [
                ['min' => 90, 'grade' => 'A+'],
                ['min' => 80, 'grade' => 'A'],
                ['min' => 70, 'grade' => 'B+'],
                ['min' => 60, 'grade' => 'B'],
                ['min' => 50, 'grade' => 'C'],
                ['min' => 40, 'grade' => 'D'],
                ['min' => 0, 'grade' => 'F'],
            ];
            
            $student_report = [
                'subjects' => [],
                'sum_gained' => 0,
                'sum_total' => 0,
                'percentage' => 0,
                'overall_grade' => 'N/A',
                'attendance_percent' => calculate_attendance($conn_core_view, $view_student_id, $class_teacher_class_id, $_GET['start_date'], $_GET['end_date'], $students_in_class_view),
                'class_id' => $class_teacher_class_id
            ];
            
            $student_pass = true;
            foreach ($required_subjects as $sub_id => $sub_name) {
                $mark_row = $marks_data[$sub_id] ?? ['total_marks' => 0, 'gained_marks' => 0, 'passing_marks' => 0];
                
                if ($mark_row['gained_marks'] < $mark_row['passing_marks']) {
                    $student_pass = false;
                }
                
                $student_report['subjects'][] = [
                    'name' => $sub_name,
                    'total_marks' => $mark_row['total_marks'],
                    'gained_marks' => $mark_row['gained_marks'],
                    'grade' => get_grade($mark_row['gained_marks'], $mark_row['total_marks'], $grade_rules),
                ];
                
                $student_report['sum_gained'] += $mark_row['gained_marks'];
                $student_report['sum_total'] += $mark_row['total_marks'];
            }
            
            if ($student_report['sum_total'] > 0) {
                $percentage = round(($student_report['sum_gained'] / $student_report['sum_total']) * 100, 2);
                $student_report['percentage'] = $percentage;
                
                if (!$student_pass) {
                    $student_report['overall_grade'] = 'F (Failed)';
                } else {
                    $student_report['overall_grade'] = get_overall_grade($percentage, $grade_rules);
                }
            }
            
            $report_data_view[$view_student_id] = [
                'id' => $view_student_id,
                'name' => $student_info['name'],
                'roll_no' => $student_info['roll_no'],
                'class_name' => $class_name,
                'exam_name' => $exam_name,
                'exam_id' => $view_exam_id,
                'details' => $student_report
            ];
        }
    }
    
    $single_report_html = '';
    if (!empty($report_data_view)) {
        $single_report_html = generate_report_card_html($report_data_view[$view_student_id], $view_school_details);
    } else {
        $single_report_html = '<div style="padding: 20px; text-align: center;">Could not load report data or database connection failed.</div>';
    }
    
    echo '<!DOCTYPE html>
    <html lang="en">
    <head>
        <meta charset="UTF-8">
        <meta name="viewport" content="width=device-width, initial-scale=1.0">
        <title>Report Card - ' . htmlspecialchars($student_info['name'] ?? 'Student') . '</title>
        <link href="https://fonts.googleapis.com/css2?family=Roboto:wght@400;500;700&display=swap" rel="stylesheet">
        <style>
            * {
                margin: 0;
                padding: 0;
                box-sizing: border-box;
            }
            
            body {
                font-family: \'Roboto\', sans-serif;
                background-color: #f4f7f6;
                padding: 20px;
            }
            
            .container {
                max-width: 900px;
                margin: 0 auto;
                background: #fff;
                padding: 30px;
                border-radius: 8px;
                box-shadow: 0 4px 12px rgba(0,0,0,0.1);
            }
            
            /* Report Card Styles */
            .report-card {
                border: 3px solid #000;
                padding: 30px;
                background: white;
                margin: 20px 0;
            }
            
            .report-header {
                /* MODIFIED: Changed to column for centered logo */
                display: flex;
                flex-direction: column; 
                align-items: center;
                gap: 10px; /* Reduced gap */
                margin-bottom: 15px;
            }
            
            .school-logo {
                width: 80px;
                height: 80px;
                object-fit: contain;
                /* flex-shrink: 0; REMOVED as flex is column now */
            }
            
            .school-info {
                flex: 1;
                text-align: center;
                width: 100%; /* Ensure text block is full width for center alignment */
            }
            
            .school-info h3 {
                font-size: 1.5rem;
                font-weight: 700;
                margin-bottom: 5px;
            }
            
            .school-info h4 {
                font-size: 1rem;
                font-weight: 700;
                text-transform: uppercase;
            }
            
            .header-separator {
                height: 3px;
                background-color: #000;
                margin: 20px 0;
            }
            
            .report-details {
                border: 1px solid #000;
                padding: 15px;
                margin-bottom: 20px;
                display: grid;
                grid-template-columns: 1fr 1fr;
                gap: 12px;
                font-size: 0.95rem;
            }
            
            .detail-item {
                line-height: 1.6;
            }
            
            .report-table {
                width: 100%;
                border-collapse: collapse;
                margin-bottom: 0;
            }
            
            .report-table th,
            .report-table td {
                border: 1px solid #000;
                padding: 12px;
                text-align: center;
                font-size: 0.95rem;
            }
            
            .report-table thead th {
                background-color: #b3d9ff;
                color: #000;
                font-weight: 700;
            }
            
            .subject-col,
            .subject-name {
                text-align: left !important;
            }
            
            .grade-cell {
                font-weight: 700;
            }
            
            .total-row {
                background-color: #b3ffcc;
                font-weight: 700;
            }
            
            .percentage-row-table td {
                background-color: #fff;
                text-align: left !important;
                padding: 12px;
                font-size: 0.95rem;
            }
            
            .percentage-cell {
                text-align: left !important;
            }
            
            .signature-section {
                margin-top: 60px;
                display: flex;
                justify-content: space-between;
                align-items: flex-end;
            }
            
            .signature-box {
                width: 45%;
            }
            
            .parent-signature {
                text-align: left;
            }
            
            .principal-signature {
                text-align: right;
            }
            
            .signature-img {
                max-width: 150px;
                max-height: 50px;
                object-fit: contain;
                margin-bottom: 5px;
                display: inline-block;
            }
            
            .signature-line {
                border-top: 1px solid #000;
                width: 100%;
                margin-bottom: 5px;
            }
            
            .signature-label {
                font-size: 0.9rem;
            }
            
            .download-btn {
                background-color: #007bff;
                color: white;
                padding: 12px 24px;
                border: none;
                border-radius: 5px;
                cursor: pointer;
                font-size: 1rem;
                margin-bottom: 20px;
                display: block;
                width: fit-content;
                margin: 0 auto 20px auto;
                font-weight: 500;
            }
            
            .download-btn:hover {
                background-color: #0056b3;
            }
            
            /* Mobile Responsive - Same layout as desktop */
            @media (max-width: 768px) {
                body {
                    padding: 10px;
                }
                
                .container {
                    padding: 20px;
                }
                
                .report-card {
                    padding: 20px;
                    border-width: 2px;
                }
                
                .report-header {
                    gap: 8px; /* Adjusted gap */
                }
                
                .school-logo {
                    width: 60px;
                    height: 60px;
                }
                
                .school-info h3 {
                    font-size: 1.2rem;
                }
                
                .school-info h4 {
                    font-size: 0.85rem;
                }
                
                .header-separator {
                    margin: 15px 0;
                    height: 2px;
                }
                
                /* Keep 2-column grid on mobile */
                .report-details {
                    grid-template-columns: 1fr 1fr;
                    gap: 8px;
                    padding: 12px;
                    font-size: 0.8rem;
                }
                
                .report-table th,
                .report-table td {
                    padding: 8px 4px;
                    font-size: 0.75rem;
                }
                
                .percentage-row-table td {
                    font-size: 0.8rem;
                    padding: 10px;
                }
                
                /* Keep horizontal signature layout on mobile */
                .signature-section {
                    margin-top: 40px;
                    gap: 10px;
                }
                
                .signature-box {
                    width: 48%;
                }
                
                .signature-label {
                    font-size: 0.75rem;
                }
                
                .download-btn {
                    width: 100%;
                    font-size: 0.9rem;
                }
            }
            
            /* Print Styles - MODIFIED to reduce headers/footers */
            @page {
                size: A4;
                margin: 0; /* Removing print margin */
            }
            
            @media print {
                body {
                    background: white;
                    padding: 0;
                    margin: 0; /* Removing body margin */
                }
                
                .container {
                    box-shadow: none;
                    padding: 0;
                    max-width: 100%;
                }
                
                .download-btn {
                    display: none !important;
                }
                
                .report-card {
                    border: 3px solid #000;
                    padding: 20px;
                    page-break-after: always;
                }
                
                .signature-section {
                    margin-top: 60px;
                }
            }
        </style>
    </head>
    <body>
        <div class="container">
            <button class="download-btn" onclick="window.print()">
                Download/Print Report Card (Save as PDF)
            </button>
            ' . $single_report_html . '
        </div>
    </body>
    </html>';
    
    end_single_view:
    if ($conn_core_view) $conn_core_view->close();
    if ($conn_teach_view) $conn_teach_view->close();
    exit;
}

// ======================================================================
// 8. HTML INTERFACE AND REPORT OUTPUT
// ======================================================================
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Class Teacher Report Card Generator</title>
    <link href="https://fonts.googleapis.com/css2?family=Roboto:wght@400;500;700&display=swap" rel="stylesheet">
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: 'Roboto', sans-serif;
            background-color: #f4f7f6;
            padding: 10px;
        }

        .container {
            max-width: 900px;
            margin: 0 auto;
            background: #fff;
            padding: 20px;
            border-radius: 8px;
            box-shadow: 0 4px 12px rgba(0,0,0,0.1);
        }

        h1 {
            color: #1a56db;
            border-bottom: 2px solid #1a56db;
            padding-bottom: 10px;
            margin-bottom: 20px;
            font-size: 1.5rem;
        }

        .alert-error {
            color: #842029;
            background-color: #f8d7da;
            border: 1px solid #f5c6cb;
            padding: 10px;
            border-radius: 5px;
            margin-bottom: 20px;
            font-size: 0.9rem;
            word-wrap: break-word;
        }

        .alert-success {
            color: #0f5132;
            background-color: #d4edda;
            border: 1px solid #c3e6cb;
            padding: 10px;
            border-radius: 5px;
            margin-bottom: 20px;
            font-size: 0.9rem;
            word-wrap: break-word;
        }

        .form-group {
            margin-bottom: 15px;
        }

        label {
            display: block;
            font-weight: 600;
            margin-bottom: 5px;
            font-size: 0.9rem;
        }

        select, input[type="date"] {
            width: 100%;
            padding: 10px;
            border: 1px solid #ccc;
            border-radius: 4px;
            box-sizing: border-box;
            font-size: 0.9rem;
        }

        .btn-group {
            display: flex;
            flex-wrap: wrap;
            gap: 10px;
            margin-top: 10px;
        }

        .btn-group button {
            flex: 1 1 auto;
            min-width: 150px;
            padding: 12px 15px;
            font-size: 0.9rem;
            border-radius: 4px;
            cursor: pointer;
            border: none;
            transition: background-color 0.3s;
            white-space: normal;
            font-weight: 500;
        }

        .btn-generate { background-color: #28a745; color: white; }
        .btn-save { background-color: #007bff; color: white; }
        .btn-print { background-color: #6c757d; color: white; }

        .btn-generate:hover { background-color: #1e7e34; }
        .btn-save:hover { background-color: #0056b3; }
        .btn-print:hover { background-color: #5a6268; }

        /* Report Card Styles - CONSISTENT across desktop and mobile */
        .report-card {
            border: 3px solid #000;
            padding: 30px;
            margin-top: 30px;
            page-break-after: always;
            background: white;
        }

        .report-header {
            /* MODIFIED: Changed to column for centered logo */
            display: flex;
            flex-direction: column;
            align-items: center;
            gap: 10px; /* Reduced gap */
            margin-bottom: 15px;
        }

        .school-logo {
            width: 80px;
            height: 80px;
            object-fit: contain;
            flex-shrink: 0;
        }

        .school-info {
            flex: 1;
            text-align: center;
            width: 100%; /* Ensure text block is full width for center alignment */
        }

        .school-info h3 {
            font-size: 1.5rem;
            font-weight: 700;
            margin-bottom: 5px;
        }

        .school-info h4 {
            font-size: 1rem;
            font-weight: 700;
            text-transform: uppercase;
        }

        .header-separator {
            height: 3px;
            background-color: #000;
            margin: 20px 0;
        }

        .report-details {
            border: 1px solid #000;
            padding: 15px;
            margin-bottom: 20px;
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 12px;
            font-size: 0.95rem;
        }

        .detail-item {
            line-height: 1.6;
        }

        .report-table {
            width: 100%;
            border-collapse: collapse;
            margin-bottom: 0;
        }

        .report-table th,
        .report-table td {
            border: 1px solid #000;
            padding: 12px;
            text-align: center;
            font-size: 0.95rem;
        }

        .report-table thead th {
            background-color: #b3d9ff;
            color: #000;
            font-weight: 700;
        }

        .subject-col,
        .subject-name {
            text-align: left !important;
        }

        .grade-cell {
            font-weight: 700;
        }

        .total-row {
            background-color: #b3ffcc;
            font-weight: 700;
        }

        .percentage-row-table td {
            background-color: #fff;
            text-align: left !important;
            padding: 12px;
            font-size: 0.95rem;
        }

        .percentage-cell {
            text-align: left !important;
        }

        .signature-section {
            margin-top: 60px;
            display: flex;
            justify-content: space-between;
            align-items: flex-end;
        }

        .signature-box {
            width: 45%;
        }

        .parent-signature {
            text-align: left;
        }

        .principal-signature {
            text-align: right;
        }

        .signature-img {
            max-width: 150px;
            max-height: 50px;
            object-fit: contain;
            margin-bottom: 5px;
            display: inline-block;
        }

        .signature-line {
            border-top: 1px solid #000;
            width: 100%;
            margin-bottom: 5px;
        }

        .signature-label {
            font-size: 0.9rem;
        }

        #pdf-list {
            margin-top: 30px;
            font-size: 1.2rem;
            font-weight: 700;
        }

        .pdf-entry {
            padding: 10px;
            border: 1px solid #eee;
            margin-bottom: 5px;
            display: flex;
            flex-wrap: nowrap; /* MODIFIED: Keeps content on one line for desktop */
            justify-content: space-between;
            align-items: center;
            background-color: #fcfcfc;
            border-radius: 4px;
            font-size: 0.9rem;
            gap: 10px;
        }

        .pdf-entry span {
            /* MODIFIED: Allows the student name/log info to take space, but not fully wrap */
            flex-grow: 1;
            flex-shrink: 1;
            flex-basis: auto;
            margin-bottom: 0; 
            word-wrap: break-word;
        }

        .pdf-entry div {
            /* MODIFIED: Keeps buttons together and prevents wrapping student info */
            flex-shrink: 0;
            display: flex;
            gap: 5px;
        }
        
        .pdf-entry div button {
            margin-left: 0 !important; /* Overriding previous margin-left */
        }

        /* Mobile Responsive - SAME alignment as desktop */
        @media (max-width: 768px) {
            body {
                padding: 5px;
            }
            
            .container {
                padding: 15px;
                border-radius: 4px;
            }
            
            h1 {
                font-size: 1.2rem;
                padding-bottom: 8px;
                margin-bottom: 15px;
            }
            
            .alert-error,
            .alert-success {
                font-size: 0.85rem;
                padding: 8px;
            }
            
            label {
                font-size: 0.85rem;
            }
            
            select, input[type="date"] {
                font-size: 0.85rem;
                padding: 8px;
            }
            
            .btn-group {
                flex-direction: column;
                gap: 8px;
            }
            
            .btn-group button {
                width: 100%;
                min-width: unset;
                font-size: 0.85rem;
                padding: 10px 15px;
            }
            
            .report-card {
                padding: 20px;
                margin-top: 20px;
                border-width: 2px;
            }
            
            .report-header {
                gap: 8px; /* Adjusted gap */
            }
            
            .school-logo {
                width: 60px;
                height: 60px;
            }
            
            .school-info h3 {
                font-size: 1.2rem;
            }
            
            .school-info h4 {
                font-size: 0.85rem;
            }
            
            .header-separator {
                margin: 15px 0;
                height: 2px;
            }
            
            /* Keep 2-column grid on mobile for consistency */
            .report-details {
                grid-template-columns: 1fr 1fr;
                gap: 8px;
                padding: 12px;
                font-size: 0.8rem;
            }
            
            .report-table th,
            .report-table td {
                padding: 8px 4px;
                font-size: 0.75rem;
            }
            
            .percentage-row-table td {
                font-size: 0.8rem;
                padding: 10px;
            }
            
            /* Keep horizontal signature layout on mobile */
            .signature-section {
                margin-top: 40px;
                gap: 10px;
            }
            
            .signature-box {
                width: 48%;
            }
            
            .signature-label {
                font-size: 0.75rem;
            }
            
            #pdf-list {
                font-size: 1rem;
                margin-top: 20px;
            }
            
            .pdf-entry {
                /* MODIFIED: Change back to column for mobile */
                flex-direction: column;
                align-items: stretch;
                padding: 12px;
                font-size: 0.85rem;
                flex-wrap: wrap; 
            }
            
            .pdf-entry span {
                /* MODIFIED */
                margin-bottom: 10px;
                flex-basis: 100%;
            }
            
            .pdf-entry div {
                /* MODIFIED */
                display: flex;
                flex-direction: column;
                gap: 8px;
                width: 100%;
            }
            
            .pdf-entry div button {
                margin-left: 0;
                width: 100%;
                padding: 10px;
                font-size: 0.85rem;
            }
        }

        /* Print Styles - MODIFIED to reduce headers/footers */
        @page {
            size: A4;
            margin: 0; /* Removing print margin */
        }

        @media print {
            .hide-on-print {
                display: none !important;
            }
            
            body {
                background: #fff;
                margin: 0; /* Removing body margin */
                padding: 0;
            }
            
            .container, .print-area {
                width: 100%;
                padding: 0;
                margin: 0;
                box-shadow: none;
            }
            
            .report-card {
                border: 3px solid #000;
                padding: 20px;
                margin: 0;
                page-break-after: always;
            }
            
            .signature-section {
                margin-top: 60px;
            }
        }
    </style>
</head>
<body>

<div class="container">
    <h1 class="hide-on-print">Class Teacher Report Generator</h1>
    
    <?php if ($error): ?>
        <div class="alert alert-error hide-on-print">❌ **Error:** <?= $error ?></div>
    <?php endif; ?>
    <?php if ($message): ?>
        <div class="alert alert-success hide-on-print">🔔 **Status:** <?= $message ?></div>
    <?php endif; ?>
    
    <?php if (!isset($class_teacher_class_id) || $error): ?>
        <p class="hide-on-print">Could not initialize report tools. Please check your system configuration or contact admin.</p>
    <?php else: ?>
        
        <form method="POST" action="<?= htmlspecialchars($_SERVER['PHP_SELF']) ?>" class="hide-on-print">
            <input type="hidden" name="class_id" value="<?= $class_teacher_class_id ?>">
            
            <div class="form-group">
                <label for="exam_id">Select Exam (Class: **<?= htmlspecialchars($class_name) ?>**):</label>
                <select name="exam_id" id="exam_id" required>
                    <option value="">-- Select Exam --</option>
                    <?php foreach ($available_exams as $id => $name): ?>
                        <option value="<?= $id ?>" <?= ($exam_id == $id) ? 'selected' : '' ?>>
                            <?= htmlspecialchars($name) ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
            
            <div style="display: flex; gap: 20px;">
                <div class="form-group" style="flex: 1;">
                    <label for="attendance_start_date">Attendance Start Date:</label>
                    <input type="date" id="attendance_start_date" name="attendance_start_date" value="<?= $start_date ?? date('Y-m-01') ?>" required>
                </div>
                <div class="form-group" style="flex: 1;">
                    <label for="attendance_end_date">Attendance End Date:</label>
                    <input type="date" id="attendance_end_date" name="attendance_end_date" value="<?= $end_date ?? date('Y-m-d') ?>" required>
                </div>
            </div>
            
            <div class="btn-group">
                <button type="submit" name="generate_report" class="btn-generate">
                    1. Preview Report / Check Completion
                </button>
                
                <?php if ($is_report_ready): ?>
                    <button type="submit" name="save_pdfs" class="btn-save">
                        2. Log Record (Ready for Print)
                    </button>
                    <button type="button" class="btn-print" onclick="window.print()">
                        3. Print/Download ALL Previewed Reports (Browser)
                    </button>
                <?php endif; ?>
            </div>
            
        </form>
    <?php endif; ?>
    
    <?php if ($class_id && $exam_id && !empty($generated_pdfs)): ?>
        <h2 id="pdf-list" class="hide-on-print">📁 Logged Report Records (<?= htmlspecialchars($available_exams[$exam_id] ?? 'N/A') ?>)</h2>
        <?php foreach ($students_in_class as $student): ?>
            <?php if (isset($generated_pdfs[$student['id']])):
                $pdf_data = $generated_pdfs[$student['id']];
                $roll_no = htmlspecialchars($student['roll_no']);
                $student_name = htmlspecialchars($student['name']);
            ?>
                <div class="pdf-entry hide-on-print">
                    <span>
                        **Roll <?= $roll_no ?>:** <?= $student_name ?>
                        (Logged on <?= date('Y-m-d H:i', strtotime($pdf_data['created_at'])) ?>)
                    </span>
                    <div style="flex-basis: auto;">
                        <button type="button"
                                class="btn-save"
                                style="background-color: #007bff; padding: 5px 10px; font-size: 0.85rem;"
                                onclick="window.open(
                                    '?view_report=1&student_id=<?= $student['id'] ?>&exam_id=<?= $exam_id ?>&start_date=<?= $start_date ?>&end_date=<?= $end_date ?>',
                                    '_blank'
                                )">
                            View Report
                        </button>
                        
                        <form method="POST" style="display: inline-block;">
                            <input type="hidden" name="exam_id" value="<?= $exam_id ?>">
                            <input type="hidden" name="attendance_start_date" value="<?= $start_date ?>">
                            <input type="hidden" name="attendance_end_date" value="<?= $end_date ?>">
                            <input type="hidden" name="regenerate_student_id" value="<?= $student['id'] ?>">
                            <button type="submit" name="save_pdfs" class="btn-save" style="background-color: #ffc107; color: #333; padding: 5px 10px; font-size: 0.85rem;">Re-log Data</button>
                        </form>
                    </div>
                </div>
            <?php endif; ?>
        <?php endforeach; ?>
    <?php endif; ?>
    
    <?php if ($is_report_ready && !empty($report_data)): ?>
        <div class="alert alert-success hide-on-print">
            ✅ Preview successful for <?= count($report_data) ?> students. If records are logged, use the **'Print/Download ALL Previewed Reports'** button now.
        </div>
        
        <div class="print-area">
        <?php
        foreach ($report_data as $student_id => $data) {
            $html_output = generate_report_card_html($data, $school_details);
            echo $html_output;
        }
        ?>
        </div>
    <?php endif; ?>

</div>

</body>
</html>
<?php

// Close connections opened in the main script block
if ($conn_teach) $conn_teach->close();
if ($conn_core) $conn_core->close();
?>